/*
 * Copyright 1993-2010 NVIDIA Corporation.  All rights reserved.
 *
 * NVIDIA Corporation and its licensors retain all intellectual property and 
 * proprietary rights in and to this software and related documentation. 
 * Any use, reproduction, disclosure, or distribution of this software 
 * and related documentation without an express license agreement from
 * NVIDIA Corporation is strictly prohibited.
 */

/* 
 * nvThread.h
 * Encapsulates all functionality for threading and events
 * Currently on Windows only
 */

//brief handle for opaque thread objects.
typedef struct nvThreadRec *nvThread;


/**
 * \brief Creates a thread object that begins execution at the
 *        specified function.
 *
 * \param threadFunc Address the new thread should begin execution at.
 *
 * \param threadData Passed as the only argument to threadFunc when it is run.
 *
 * \return A new thread handle on success.
 *         NULL on failure
 */
extern nvThread nvCreateThread(unsigned (*threadFunc)(void *),
                                   void *threadData);

/**
 * \brief Waits for the specified thread to finish execution and then
 *        cleans up any resources associated with it.
 *
 * \param thread The thread handle to tear down.
 *
 * \param threadReturn The value returned by the thread's threadFunc.
 *
 * \return A non-zero value on success
 *         Zero on failure.
 */
extern int nvWaitForAndDestroyThread(nvThread thread,
                                       unsigned int *threadReturn);

/*@}*/


/**
 * \defgroup Events Event objects
 *
 * These objects can be used to efficiently wait for events to
 * occur.  The event objects here mimic win32 auto-reseting
 * events.
 */

/*@{*/

/**
 * \brief handle for opaque event object.
 *
 * These behave like win32 auto-reset event objects.
 */
typedef struct nvEventRec *nvEvent;

/**
 * \brief Creates an auto-resetting event object.  Initially the event
 *        is in the not signaled state.
 *
 * \return An event handle on success.
 *         NULL on failure.
 */
extern nvEvent nvCreateEvent(void);

/**
 * \brief Destroys an event created with nvCreateEvent().
 *
 * \param event An event handle returned by nvCreateEvent().
 */
extern void nvDestroyEvent(nvEvent event);

/**
 * \brief Waits for an event to reach the signaled state.
 *
 * \param event An event handle returned by nvCreateEvent().
 *
 * \return non-zero on success.
 *         zero on failure.
 */
extern int nvWaitForEvent(nvEvent event);

/**
 * \brief Puts an event in the signaled state and wakes up any
 *        threads waiting for the event.
 *
 * \param event An event handle that is NOT in the signaled state.
 *
 * \return non-zero on success.
 *         zero on failure.
 */
extern int nvSignalEvent(nvEvent event);

/*@}*/

/*@}*/

