/* ---------------------------------------------------------------------------
 * This software is in the public domain, furnished "as is", without technical
 * support, and with no warranty, express or implied, as to its usefulness for
 * any purpose.

 * Author: Wil Braithwaite.
 *
 */

#ifndef EASY_CUBUFFER_H_INCLUDED
#define EASY_CUBUFFER_H_INCLUDED

#include "std_utils.h"
#include <ostream>

namespace Easy
{
namespace Cu
{

template<class T>
class BufferMapper;

class Buffer
{
public:
    static int debugging;
    static const int MAX_PAGES;
    enum MemType {NONE, HOST, VBO, CUDA};
    char name[64];
protected:
    mutable bool owner;

public:

    static const char* typeNames[];

    class CopyOptions
    {
    public:
        CopyOptions();
        CopyOptions& SetStream(long v);
        friend std::ostream &operator<< (std::ostream &out, const CopyOptions &v);
    private:
        long cuStream;
        friend class Buffer;
    };

    enum {MAP_ACCESS_READ_WRITE=0, MAP_ACCESS_READ_ONLY=1, MAP_ACCESS_WRITE_ONLY=2};

    class MapOptions
    {
    public:
        MapOptions();
        MapOptions& SetAccess(int v);
        MapOptions& SetStream(long v);
        friend std::ostream &operator<< (std::ostream &out, const MapOptions &v);
    private:
        int access;
        long cuStream;
        friend class Buffer;
    };

    class ClearOptions
    {
    public:
        ClearOptions();
        ClearOptions& SetStream(long v);
        friend std::ostream &operator<< (std::ostream &out, const ClearOptions &v);
    private:
        long cuStream;
        friend class Buffer;
    };

    static const CopyOptions DefaultCopyOptions;
    static const MapOptions DefaultMapOptions;
    static const ClearOptions DefaultClearOptions;

protected:

    class BufferData
    {
    public:
        MemType type;
        bool pinned;
        Buffer *mapped_from;//, *mapped_to;
        MapOptions mapOptions;
        union
        {
            unsigned int vboId;
            void *buffer;
        };
        unsigned int vboTarget; // (GLenum) GL_ARRAY_BUFFER, GL_PIXEL_PACK_BUFFER, etc.
        unsigned int vboUsage;  // (GLenum) GL_STREAM_DRAW, GL_STATIC_READ, etc.
        mutable void *graphicsResource;
        size_t size;
        bool isExternal;

        bool operator==(const BufferData& s) const;

    } data;

public:

    Buffer(const Buffer &s);
    Buffer(MemType type=NONE, size_t size=0, void *buf=0, bool owner=false);
    ~Buffer();

    Buffer &operator=(const Buffer &);
    //Buffer &operator=(Buffer &);

    BufferData release() const;

    Buffer &Map(const Buffer &source, const MapOptions& options=DefaultMapOptions);
    bool Unmap();

    size_t Size() const
    {
        return data.size;
    }

    enum MemType Type() const
    {
        return data.type;
    }

    const char* Name() const
    {
        return name;
    }

    std::string ShortName() const;

    void *Data() const;
    unsigned int Vbo() const;

    bool IsMapTarget()
    {
        return (data.mapped_from!=0);
    }

    bool Allocate(MemType type, size_t size, int flags=0, const char* _name=0);
    bool Free();
    size_t Copy(const Buffer &source, size_t d_offset=0, size_t s_offset=0, size_t count=size_t(-1), const CopyOptions& options=DefaultCopyOptions);
    bool Clear(int v=0, const ClearOptions& options=DefaultClearOptions);
    bool Dump(const char* title=0, size_t n=size_t(-1), size_t step=1) const;

    template<class T> inline bool DumpAs(const char* title=0, size_t count=size_t(-1), size_t step=1) const
    {
        if (data.type == HOST)
        {
            if (title)
                std::cerr << title << std::endl;
            std::cerr << *this << std::endl;
            if(count == size_t(-1))
                count = data.size/sizeof(T);
            else if(count > data.size/sizeof(T))
                count = data.size/sizeof(T);
			if(step == 0)
				step = 1;
            for (size_t i=0; i<count; i+=step)
                std::cerr << i << ". " << ((T*)Data())[i] << std::endl;
            return true;
        }
        else
        {
            Buffer h_mem(HOST, Size());
            h_mem.Copy(*this);
            return h_mem.DumpAs<T>(title, count, step);
        }
    }

    bool Valid() const
    {
        return (data.size > 0);
    }

    enum AllocHostFlags {HOST_PINNED=0x01, HOST_MAPPABLE=0x02, HOST_WRITE_COMBINED=0x04, HOST_PORTABLE=0x08};

    friend std::ostream &operator<< (std::ostream &out, const Buffer &v);

protected:
    void Init();

    bool AllocateHost(size_t size, int flags=0);
    void FreeHost();
    bool CreateMapFromHostTo(Buffer &d, const MapOptions& options) const;
    static void DeleteMapFromHostTo(Buffer &d);
    bool CopyToHostFrom(const Buffer &source, size_t d_offset, size_t s_offset, size_t count, const CopyOptions& options);

    bool AllocateCUDA(size_t size);
    void FreeDevice();
    bool CreateMapFromDeviceTo(Buffer &d, const MapOptions& options) const;
    static void DeleteMapFromCudaTo(Buffer &d);
    bool CopyToDeviceFrom(const Buffer &source, size_t d_offset, size_t s_offset, size_t count, const CopyOptions& options);

    bool AllocateVBO(size_t size, unsigned int target, unsigned int vboId=0, int flags=0, const char* _name=0);
    void FreeVBO();
    bool CreateMapFromVBOTo(Buffer &d, const MapOptions& options) const;
    static void DeleteMapFromVBOTo(Buffer &d);
    bool CopyToVBOFrom(const Buffer &source, size_t d_offset, size_t s_offset, size_t count, const CopyOptions& options);


    template<class T>
    friend class BufferMapper;
};

template<class T>
class BufferMapper
{
public:
    class Buffer internal;
    bool valid;

    BufferMapper(Buffer::MemType type)
        :
        valid(false),
        internal(type)
    {
    }

    BufferMapper(Buffer::MemType type, Buffer &s, const Buffer::MapOptions& options=Buffer::DefaultMapOptions)
        :
        internal(type)
    {
        internal.Map(s, options);
        valid = (internal.Size() > 0);
    }

    ~BufferMapper()
    {
        /*if(valid)
        {
            if(!internal.IsMapTarget())
            {
                // we didn't map so instead we must transfer the data
                mem.Copy(internal);
            }
        }*/
    }

    inline bool Map(const Buffer &s, const Buffer::MapOptions& options=Buffer::DefaultMapOptions)
    {
        internal.Map(s, options);
        valid = (internal.Size() > 0);
        return valid;
    }

    inline bool Unmap()
    {
        bool rc = internal.Unmap();
        internal.data.size = 0;
        valid = false;
        return rc;
    }

    inline bool Valid() const
    {
        return valid;
    }

    inline operator T*() const;

    inline T* Pointer() const;

    const Buffer& operator*() const;
    const Buffer* operator->() const;

    bool Dump(const char* title=NULL, int count=-1) const;

    friend std::ostream &operator<< (std::ostream &s, const BufferMapper &v)
    {
        if(v.valid)
            s << "BufferMapper: " << v.internal;
        else
            s << "BufferMapper: invalid";
        return s;
    }
};

//----------------------------------------------------------------------------------------------
template<class T>
inline T* BufferMapper<T>::Pointer() const
{
    if(valid)
        return (T *)internal.Data();
    return NULL;
}

//----------------------------------------------------------------------------------------------
template<class T>
inline BufferMapper<T>::operator T*() const
{
    return Pointer();
}

//----------------------------------------------------------------------------------------------
template<class T>
inline const Buffer& BufferMapper<T>::operator*() const
{
    return internal;
}

//----------------------------------------------------------------------------------------------
template<class T>
inline const Buffer* BufferMapper<T>::operator->() const
{
    return &internal;
}

//----------------------------------------------------------------------------------------------
template<class T>
bool BufferMapper<T>::Dump(const char* title, int count) const
{
    return internal.DumpAs<T>(title, count);
}

//------------------------------------------------------------------------------------------
}
}

#endif // CUBUFFER_H_INCLUDED
