/* ---------------------------------------------------------------------------
 * This software is in the public domain, furnished "as is", without technical
 * support, and with no warranty, express or implied, as to its usefulness for
 * any purpose.

 * Author: Wil Braithwaite.
 *
 */

#include "GlutApp.h"

#include "gl_utils.h"
#include "std_utils.h"
#include "std_utils.h"
#include "Profiler.h"
#include <algorithm>
#include <cmath>

#undef DrawText

namespace Easy
{


float cameraTranslateScale = 10;

// view params
int GlutApp::ox=0, GlutApp::oy=0;
int GlutApp::buttonState=0;
unsigned int GlutApp::width=800;
unsigned int GlutApp::height=800;
unsigned int GlutApp::fov=60;
bool GlutApp::using_idle = true;
GlutApp *GlutApp::app = NULL;
bool GlutApp::paused = false;
bool GlutApp::draw_profiler = true;
bool GlutApp::_exitComplete = false;

Timer GlutApp::timer;

static float camera_trans[] = {0, -80, -200};
static float camera_rot[]   = {0, 0.05, 0};
static float camera_trans_lag[] = {0, -80, -200};
static float camera_rot_lag[] = {0, 0, 0};
static const float inertia = 5;


static bool displaySliders = true;
static bool demoMode = false;
static int idleCounter = 0;
//static int demoCounter = 0;
static const int idleDelay = 2000;

static float modelView[16];

#ifdef _WIN32
// This is specifically to enable the application to enable/disable vsync
typedef BOOL (WINAPI *PFNWGLSWAPINTERVALFARPROC)( int );

void setVSync(int interval)
{
	if(WGL_EXT_swap_control)
	{
		wglSwapIntervalEXT = (PFNWGLSWAPINTERVALFARPROC)wglGetProcAddress( "wglSwapIntervalEXT" );
		wglSwapIntervalEXT(interval);
	}
}
#endif

void GlutApp::SetTitle(const char* title)
{
    glutSetWindowTitle(Stringf("%s - (%d fps) %s" ,GlutApp::app->Title(), int(ifps), title).c_str());
}

float GlutApp::computeFPS(unsigned int fpsLimit)
{
    static unsigned int frameCount = 0;
    static unsigned int fpsCount = 0;
    static float ifps = 0.f;

    frameCount++;
    fpsCount++;
    if (fpsCount == fpsLimit)
    {
		float t = timer.Value();
        ifps = fpsLimit / t;
        fpsCount = 0;
        timer.Reset();

        glutSetWindowTitle(Stringf("%s - (%d fps)" ,GlutApp::app->Title(), int(ifps)).c_str());
    }
    return ifps;
}

static void initGL(int argc, char **argv, int mode)
{
    glutInit(&argc, argv);
    glutInitDisplayMode(mode);
    glutInitWindowSize(GlutApp::width, GlutApp::height);
    glutCreateWindow(GlutApp::app->Title());

    if(glewInit() != GLEW_OK)
    {
        fprintf(stderr, "Unable to initialize GLEW.");
        exit(-1);
    }

    if (!glewIsSupported("GL_VERSION_2_0 GL_VERSION_1_5 GL_ARB_multitexture GL_ARB_vertex_buffer_object"))
    {
        fprintf(stderr, "Required OpenGL extensions missing.");
        exit(-1);
    }

#if defined (_WIN32)
    if (wglewIsSupported("WGL_EXT_swap_control"))
    {
        // disable vertical sync
        wglSwapIntervalEXT(0);
    }
#endif

    glEnable(GL_DEPTH_TEST);
    glClearColor(0, 0, 0, 0);

    glutReportErrors();
}

void GlutApp::_display()
{
	glCheckErrors();

    int w = GlutApp::app->width;
    int h = GlutApp::app->height;

    glClearColor(0.3,0.3,0.3,0);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glEnable(GL_DEPTH_TEST);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(GlutApp::app->fov,float(w)/float(h),0.1,1000);

    // view transform
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    if (GlutApp::using_idle)
    {
        for (int c = 0; c < 3; ++c)
        {
            camera_trans_lag[c] += (camera_trans[c] - camera_trans_lag[c]) * inertia;
            camera_rot_lag[c] += (camera_rot[c] - camera_rot_lag[c]) * inertia;
        }
        glTranslatef(camera_trans_lag[0], camera_trans_lag[1], camera_trans_lag[2]);
        glRotatef(camera_rot_lag[0], 1.0, 0.0, 0.0);
        glRotatef(camera_rot_lag[1], 0.0, 1.0, 0.0);
    }
    else
    {
        glTranslatef(camera_trans[0], camera_trans[1], camera_trans[2]);
        glRotatef(camera_rot[0], 1.0, 0.0, 0.0);
        glRotatef(camera_rot[1], 0.0, 1.0, 0.0);
    }

    glGetFloatv(GL_MODELVIEW_MATRIX, modelView);

    Easy::gl::drawColorAxes();

    glEnable(GL_LIGHTING);

    GlutApp::app->OnRender();

    // draw some axes!
    glDisable(GL_LIGHTING);

    // draw an axis over the screen
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    glOrtho(0,w,0,h,-1,1);
    glMatrixMode(GL_MODELVIEW);

    float matrot[16];
    memset(matrot,0,16*sizeof(float));
    memcpy(matrot,modelView,12*sizeof(float));
    matrot[15] = 1;

    glPushMatrix();
    glLoadIdentity();
    glScalef(1,1,0.001);
    glTranslatef(w-64,h-64,0);
    glScalef(48,48,48);
    glMultMatrixf(matrot);

    Easy::gl::drawColorAxes();

    glLoadIdentity();

    glDisable(GL_DEPTH_TEST);
    Easy::gl::glColorABGR(0xffffffff);
    GlutApp::app->OnOverlay();

    glPopMatrix();

    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);

    glutSwapBuffers();
    glutReportErrors();
	glCheckErrors();

    GlutApp::app->ifps = GlutApp::computeFPS(100);
}

void GlutApp::_reshape(int w, int h)
{
    glMatrixMode(GL_MODELVIEW);
    glViewport(0, 0, w, h);
    GlutApp::app->width = w;
    GlutApp::app->height = h;
    GlutApp::app->OnReshape(w, h, GlutApp::app->fov);
	glCheckErrors();
}

static void _mouse(int button, int state, int x, int y)
{
    int mods;

    if (state == GLUT_DOWN)
        GlutApp::app->buttonState |= 1<<button;
    else if (state == GLUT_UP)
        GlutApp::app->buttonState = 0;

    mods = glutGetModifiers();
    if (mods & GLUT_ACTIVE_SHIFT)
    {
        GlutApp::app->buttonState = 2;
    }
    else if (mods & GLUT_ACTIVE_CTRL)
    {
        GlutApp::app->buttonState = 3;
    }

    GlutApp::app->ox = x;
    GlutApp::app->oy = y;

    idleCounter = 0;

    glutPostRedisplay();
}

static void _motion(int x, int y)
{
    float dx, dy;
    dx = x - GlutApp::app->ox;
    dy = y - GlutApp::app->oy;

    if (GlutApp::app->buttonState == 3)
    {
        // left+middle = zoom
        camera_trans[2] += (dy / cameraTranslateScale) * 0.5f * fabs(camera_trans[2]);
    }
    else if (GlutApp::app->buttonState & 2)
    {
        // middle = translate
        camera_trans[0] += dx / cameraTranslateScale;
        camera_trans[1] -= dy / cameraTranslateScale;
    }
    else if (GlutApp::app->buttonState & 1)
    {
        // left = rotate
        camera_rot[0] += dy / 5.f;
        camera_rot[1] += dx / 5.f;
    }

    GlutApp::app->ox = x;
    GlutApp::app->oy = y;

    demoMode = false;
    idleCounter = 0;

    glutPostRedisplay();
}

static void _key(unsigned char key, int /*x*/, int /*y*/)
{
    if (!GlutApp::app->OnKey(key))
    {
        switch (key)
        {
        case 'h':
            displaySliders = !displaySliders;
            break;
        default:
            fprintf(stderr,"Unrecognized input \'%c\'\n",key);
        }
    }

    demoMode = false;
    idleCounter = 0;
    glutPostRedisplay();
}

static void _special(int k, int x, int y)
{
    demoMode = false;
    idleCounter = 0;
}

static void _idle(void)
{
    GlutApp::app->OnIdle();
    glutPostRedisplay();
}

static void _cleanup()
{
    if (!GlutApp::_exitComplete)
        GlutApp::app->OnExit();
}

static void _menu(int i)
{
    _key((unsigned char) i, 0, 0);
}

struct HotKeyOptions
{
    char text[256];
    int key;
} hotkey_options[]=
{
    {"Reset",'1'},
    {"Pause    [p]", 'p'},
    {"Profiler [g]", 'g'},
};

void initMenus()
{
    glutCreateMenu(_menu);

    glutAddMenuEntry("Quit     [esc]", '\033');
    glutAddMenuEntry("----------",0);

    for (int i=0;i<3;++i)
    {
        glutAddMenuEntry(hotkey_options[i].text, hotkey_options[i].key);
    }

    glutAttachMenu(GLUT_RIGHT_BUTTON);
}

GlutApp::GlutApp(int _argc, char **_argv, int mode)
{
    argc = _argc;
    argv = _argv;
    glut_mode = mode;
    if (app)
    {
        //printf("application instance already exists!\n");
        exit(1);
    }
    app = this;

    draw_profiler = true;
    paused = false;
}

void GlutApp::Run()
{
    initGL(argc, argv, glut_mode);

    initMenus();

    glutDisplayFunc(_display);
    glutReshapeFunc(_reshape);
    glutMouseFunc(_mouse);
    glutMotionFunc(_motion);
    glutKeyboardFunc(_key);
    glutSpecialFunc(_special);
    glutIgnoreKeyRepeat(1);

    //SetPause(false);

    atexit(_cleanup);

#ifdef _WIN32
    setVSync(0);
#endif

    OnInit();

    glutMainLoop();
}

void GlutApp::SetPause(bool state)
{
    paused = state;
    using_idle = false;
    if(paused)
        glutIdleFunc(NULL);
    else
    {
        glutIdleFunc(_idle);
        using_idle = true;
    }
}

void GlutApp::Repaint()
{
    glutPostRedisplay();
}


const char *GlutApp::Title()
{
    return "GlutApp";
}

void GlutApp::OnInit()
{
}

void GlutApp::OnRender()
{
}

void GlutApp::OnOverlay()
{
    if(paused)
    {
        glColor4f(1,1,1,1);
        Easy::gl::drawSolidRectangle(0,height-16, width, height-32-4);
        glColor4f(1,0,0,1);
        Easy::gl::drawText(width/2,height-32,"PAUSED", GLUT_BITMAP_9_BY_15);
    }
}

void GlutApp::OnUpdate()
{
}

void GlutApp::OnReshape(unsigned int w, unsigned int h, unsigned int fov)
{
}

void GlutApp::OnIdle()
{
}

bool GlutApp::OnKey(unsigned char key)
{
    switch (key)
    {
    case 'p':
        paused = !paused;
        if(paused)
            glutIdleFunc(NULL);
        else
            glutIdleFunc(_idle);
        return true;

    case '\033':
    case 'q':
        exit(0);
        return true;

    case 'g':
        draw_profiler = !draw_profiler;
        return true;
    }
    return false;
}

void GlutApp::OnExit()
{
    std::cout << "OnExit" << std::endl;
    _exitComplete = true;
}


static void recurse_profile(Profiler::ProfileIterator *iter,int global_xoffset, int &yoffset,int indent,float scale,int numFrames)
{
    int xoffset;
    for (;iter->IsDone() == false;iter->Next())
    {
        if (iter->Calls() == 0)
            continue;

        float thisscale = 1.f;
        if (iter->ParentElapsedTime() > 0.f)
            thisscale = iter->ElapsedTime()/iter->ParentElapsedTime();

		if(thisscale < 0)
			thisscale = 0;
		if(thisscale > 1)
			thisscale = 1;

        int percentage = (int)(100.f*thisscale);

        float count = numFrames;
        FORCE_MAX(count,1);

        float averageTimePerCall = float(iter->ElapsedTime()*1000.f)/iter->Calls();
        float timePerFrame = float(iter->ElapsedTime()*1000.f)/count;
        int callsPerFrame = int(iter->Calls()/count);


        const int NAME_XOFFSET=0+global_xoffset;
        const int GRAPH_XOFFSET=256+global_xoffset;
        const int INDENT_PIXELS=20;
        const int GRAPH_SIZE=30;
        const int YSPACING=16;

        xoffset = NAME_XOFFSET+indent*INDENT_PIXELS;
        Easy::gl::drawText(xoffset,yoffset,iter->Name());

        xoffset = GRAPH_XOFFSET;
        Easy::gl::drawSolidRectangle(xoffset,yoffset, xoffset+GRAPH_SIZE*(thisscale*scale),yoffset+10);

        xoffset = GRAPH_XOFFSET+GRAPH_SIZE+INDENT_PIXELS;
        std::string t = Stringf("calls=%05d time=%0.3fms avg/frame=%0.3fms -- %03d%%",callsPerFrame, timePerFrame, averageTimePerCall, percentage);
        Easy::gl::drawText(xoffset,yoffset,t.c_str());

        yoffset -= YSPACING;

        Profiler::ProfileIterator iter2(iter->ThisNode());
        recurse_profile(&iter2,global_xoffset,yoffset,indent+1,thisscale*scale,numFrames);
    }
}


void GlutApp::DrawProfile(Easy::Profiler::Manager* profiler)
{
	if(!profiler)
		return;
    Profiler::ProfileIterator *iter = profiler->CreateIterator();
	int numFrames = profiler->ElapsedFrames();

    if (!iter->IsDone())
    {
        int startx = 32;
        int starty = 48;
        int y = height-starty;
        recurse_profile(iter,startx,y,0,1,numFrames);
    }
    delete iter;
}

}


