#ifndef EASY_PROFILER_H_INCLUDED
#define EASY_PROFILER_H_INCLUDED

#include "std_utils.h"

/*
Adapted from "Real-Time Hierarchical Profiling for Game Programming Gems 3"
*/

namespace Easy
{
namespace Profiler
{

//-------------------------------------------------------------------------------------------
class Node
{
public:
    Node(const char *name="", Node *parent=0);
    virtual ~Node();

    void Setup(const char *name, Node *parent=0);

    Node *Parent() const
    {
        return parent;
    }

    Node *Sibling() const
    {
        return sibling;
    }

    Node *Child() const
    {
        return child;
    }

    void Reset();

    const char *Name(void) const
    {
        return name;
    }

    int	Calls() const
    {
        return calls;
    }

    float ElapsedTime() const
    {
        return elapsed_time;
    }

    Node *FindChild(const char * name);
    Node *AddChild(const char *name);
    Node *EnsureChild(const char *name);

    virtual void Call();
    virtual bool Return();

protected:

    const char *name;
    int	calls;
    float elapsed_time;
    float start_time;
    int	recursion_counter;

    Node *parent;
    Node *child;
    Node *sibling;

    friend class Manager;
};

//-------------------------------------------------------------------------------------------
class ProfileIterator
{
public:
    ProfileIterator(Node *start);

    // tree traversal
    void First(void);
    void Next(void);
    bool IsDone(void);

    bool EnterChild(int index);
    void EnterParent();

    // accessors for the current node

    Node *ThisNode()
    {
        return current_child;
    }

    const char *Name()
    {
        return current_child->Name();
    }

    int	Calls()
    {
        return current_child->Calls();
    }

    float ElapsedTime()
    {
        return current_child->ElapsedTime();
    }

    const char *ParentName()
    {
        return current_parent->Name();
    }

    int	ParentCalls()
    {
        return current_parent->Calls();
    }

    float ParentElapsedTime()
    {
        return current_parent->ElapsedTime();
    }

protected:

    Node *current_parent;
    Node *current_child;


    friend class Manager;
};

//-------------------------------------------------------------------------------------------
class  Manager
{
public:
	Manager();

    Node *AddNode(const char *name, Node *parent, const char *type);

    void Push(const char * name, const char *type=0);
    void Pop();

    void Reset();
    void IncrementFrame();

    int ElapsedFrames()
    {
        return frame_counter;
    }

    float ElapsedTime();

    ProfileIterator *CreateIterator()
    {
        return new ProfileIterator(&node_root);
    }

	void DumpProfile();

	void GetTicks(float* ticks);
	float GetTickRate();

    Timer timer;
protected:
    Node node_root;
private:
    Node *current_node;
    unsigned int frame_counter;
    float reset_time;
    unsigned int node_count;

public:
    Node *FindNode(const char *name, Node *root=0);
};

//-------------------------------------------------------------------------------------------
}
}

#endif // EASY_PROFILER_H_INCLUDED
