/* ---------------------------------------------------------------------------
 * This software is in the public domain, furnished "as is", without technical
 * support, and with no warranty, express or implied, as to its usefulness for
 * any purpose.

 * Author: Wil Braithwaite.
 *
 */

#include "gl_utils.h"
#include "math_utils.h"
#include <cassert>
#include <iostream>
#include <cstring>
#include <cstdio>
#include <vector>

#if defined( _WIN32 )

#else
#include <GL/glx.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

#endif

namespace Easy
{
namespace gl
{

static bool glewInitialized = false;
bool initialize()
{
    if (!glewInitialized)
    {
        glewInitialized = true;
        GLenum err = glewInit();
        if(err != GLEW_OK)
        {
            fprintf(stderr, "Glew failed: %s\n", (char *)glewGetErrorString(err));
            abort();
        }

        return true;
    }
    return true;
}

//------------------------------------------------------------------------------------------
void glColorABGR(unsigned int abgr)
{
    glColor4ub(abgr&0xff,(abgr>>8)&0xff,(abgr>>16)&0xff,(abgr>>24)&0xff);
}

//------------------------------------------------------------------------------------------
void* getContext()
{
#if defined( _WIN32 )
    return (void*)wglGetCurrentContext();
#else
    return (void*)glXGetCurrentContext();
#endif
}

//------------------------------------------------------------------------------------------
void* getDisplay()
{
#if defined( _WIN32 )
	return (void*)wglGetCurrentDC();
#else
    return (void*)glXGetCurrentDisplay();
#endif
}

//------------------------------------------------------------------------------------------
GLuint createTexture2D(GLenum target, int w, int h, GLint internalformat, GLenum format)
{
    GLuint texid;
    glGenTextures(1, &texid);
    glBindTexture(target, texid);

    glTexParameteri(target, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(target, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri(target, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(target, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

    glTexImage2D(target, 0, internalformat, w, h, 0, format, GL_FLOAT, 0);
    return texid;
}

//------------------------------------------------------------------------------------------
/// target and usage are according to OpenGK spec.
///
GLuint createBuffer(int size, GLenum target, GLenum usage)
{
	GLuint bo = 0;
    glGenBuffers(1, &bo);
    if (!glCheckErrors())
        return 0;
    if (!bo)
        return false;

    glBindBuffer(target, bo);
    glBufferData(target, size, 0, usage);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
	glCheckErrors();
    return bo;
}

//------------------------------------------------------------------------------------------
GLuint compileProgram(const char *vsource, const char *fsource, const char *gsource, GLenum in_geom_type, GLenum out_geom_type, GLint out_geom_count)
{
    GLuint program = glCreateProgram();

    if (vsource)
    {
        GLuint vertexShader = glCreateShader(GL_VERTEX_SHADER);
        glShaderSource(vertexShader, 1, &vsource, 0);
        glCompileShader(vertexShader);
        glAttachShader(program, vertexShader);
        glCheckErrors();
    }
    if (fsource)
    {
        GLuint fragmentShader = glCreateShader(GL_FRAGMENT_SHADER);
        glShaderSource(fragmentShader, 1, &fsource, 0);
        glCompileShader(fragmentShader);
        glAttachShader(program, fragmentShader);
        glCheckErrors();
    }
    if (gsource)
    {
        GLuint geometryShader = glCreateShader(GL_GEOMETRY_SHADER_EXT);
		glCheckErrors();
        glShaderSource(geometryShader, 1, (const GLchar **)&gsource, 0);
        glCompileShader(geometryShader);
        glAttachShader(program, geometryShader);
		glCheckErrors();
        glProgramParameteriEXT(program, GL_GEOMETRY_INPUT_TYPE_EXT, in_geom_type);
        glCheckErrors();
		glProgramParameteriEXT(program, GL_GEOMETRY_OUTPUT_TYPE_EXT, out_geom_type);
        glCheckErrors();
		glProgramParameteriEXT(program, GL_GEOMETRY_VERTICES_OUT_EXT, out_geom_count);
        glCheckErrors();
    }

    glLinkProgram(program);

    // check if program linked
    GLint success = 0;
    glGetProgramiv(program, GL_LINK_STATUS, &success);

    if (!success)
    {
        char temp[256];
        glGetProgramInfoLog(program, 256, 0, temp);
        printf("Failed to link program:\n%s\n", temp);
        glDeleteProgram(program);
        program = 0;
    }

    return program;
}

//------------------------------------------------------------------------------------------
void deleteBuffer(GLuint* bo)
{
    if (bo && *bo != 0)
    {
        glDeleteBuffers(1, bo);
        *bo = 0;
    }
    glCheckErrors();
}

//------------------------------------------------------------------------------------------
void drawSolidRectangle(const float lx, const float ly, const float hx, const float hy)
{
    const float p[4*2]=
    {
        lx,ly,
        hx,ly,
        hx,hy,
        lx,hy,
    };
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(2,GL_FLOAT,0,p);
    glDrawArrays(GL_TRIANGLE_FAN,0,4);
    glDisableClientState(GL_VERTEX_ARRAY);
}

//------------------------------------------------------------------------------------------
void drawLine(float* a, float* b)
{
    assert(a && b);
    glBegin(GL_LINES);
    glVertex3fv(a);
    glVertex3fv(b);
    glEnd();
}

//------------------------------------------------------------------------------------------
void drawBox(float * a, float * b)
{
    assert(a && b);
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glTranslatef(a[0],a[1],a[2]);
    glScalef(b[0]-a[0],b[1]-a[1],b[2]-a[2]);
    glTranslatef(0.5,0.5,0.5);
    drawCube(0.5);
    glPopMatrix();
}

//------------------------------------------------------------------------------------------
void drawCube(const float radius)
{
    const float verts[] =
    {
        // bottom
        -radius, -radius,  radius,
        radius, -radius,  radius,
        radius, -radius, -radius,
        -radius, -radius, -radius,
        // top
        -radius,  radius, -radius,
        radius,  radius, -radius,
        radius,  radius,  radius,
        -radius,  radius,  radius,
        // back
        -radius, -radius, -radius,
        radius, -radius, -radius,
        radius,  radius, -radius,
        -radius,  radius, -radius,
        // front
        -radius,  radius,  radius,
        radius,  radius,  radius,
        radius, -radius,  radius,
        -radius, -radius,  radius,
        // left
        -radius, -radius,  radius,
        -radius, -radius, -radius,
        -radius,  radius, -radius,
        -radius,  radius,  radius,
        // right
        radius, -radius, -radius,
        radius, -radius,  radius,
        radius,  radius,  radius,
        radius,  radius, -radius
    };

    const char vn[] =
    {
        // bottom
        0, -127,  0,
        0, -127,  0,
        0, -127,  0,
        0, -127,  0,
        // top
        0,  127,  0,
        0,  127,  0,
        0,  127,  0,
        0,  127,  0,
        // back
        0,  0, -127,
        0,  0, -127,
        0,  0, -127,
        0,  0, -127,
        // front
        0,  0,  127,
        0,  0,  127,
        0,  0,  127,
        0,  0,  127,
        // left
        -127,  0,  0,
        -127,  0,  0,
        -127,  0,  0,
        -127,  0,  0,
        // right
        127,  0,  0,
        127,  0,  0,
        127,  0,  0,
        127,  0,  0,
    };

    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_NORMAL_ARRAY);

    //glColor4f(0, 1, 0, 1);
    glNormalPointer(GL_BYTE,0,vn+0);
    glVertexPointer(3, GL_FLOAT, 0, verts);
    glDrawArrays(GL_TRIANGLE_FAN, 0, 4);

    //glColor4f(1, 0, 1, 1);
    glNormalPointer(GL_BYTE,0,vn+12);
    glVertexPointer(3, GL_FLOAT, 0, verts + 12);
    glDrawArrays(GL_TRIANGLE_FAN, 0, 4);

    //glColor4f(0, 0, 1, 1);
    glNormalPointer(GL_BYTE,0,vn+24);
    glVertexPointer(3, GL_FLOAT, 0, verts + 24);
    glDrawArrays(GL_TRIANGLE_FAN, 0, 4);

    //glColor4f(1, 1, 0, 1);
    glNormalPointer(GL_BYTE,0,vn+36);
    glVertexPointer(3, GL_FLOAT, 0, verts + 36);
    glDrawArrays(GL_TRIANGLE_FAN, 0, 4);

    //glColor4f(1, 0, 0, 1);
    glNormalPointer(GL_BYTE,0,vn+48);
    glVertexPointer(3, GL_FLOAT, 0, verts + 48);
    glDrawArrays(GL_TRIANGLE_FAN, 0, 4);

    //glColor4f(0, 1, 1, 1);
    glNormalPointer(GL_BYTE,0,vn+60);
    glVertexPointer(3, GL_FLOAT, 0, verts + 60);
    glDrawArrays(GL_TRIANGLE_FAN, 0, 4);

    glDisableClientState(GL_VERTEX_ARRAY);
    glDisableClientState(GL_NORMAL_ARRAY);

}

//------------------------------------------------------------------------------------------
void drawWireCube(float radius)
{
    const float verts[] =
    {
        // bottom
        -radius, -radius,  radius,
        radius, -radius,  radius,
        radius, -radius, -radius,
        -radius, -radius, -radius,
        // top
        -radius,  radius, radius,
        radius,  radius, radius,
        radius,  radius,  -radius,
        -radius,  radius,  -radius,
    };

    const unsigned char indices[] =
    {
        0,1, 1,2, 2,3, 3,0, 4,5, 5,6, 6,7, 7,4, 0,4, 1,5, 2,6, 3,7,
    };

    glEnableClientState(GL_VERTEX_ARRAY);

    glVertexPointer(3, GL_FLOAT, 0, verts);
    glDrawElements(GL_LINES, 24, GL_UNSIGNED_BYTE, indices);

    glDisableClientState(GL_VERTEX_ARRAY);
}

//------------------------------------------------------------------------------------------
void drawWireBox(float * a, float * b)
{
    assert(a && b);
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glTranslatef(a[0],a[1],a[2]);
    glScalef(b[0]-a[0],b[1]-a[1],b[2]-a[2]);
    glTranslatef(0.5,0.5,0.5);
    drawWireCube(0.5);
    glPopMatrix();
}

//------------------------------------------------------------------------------------------
void drawLocator(float * p, float radius)
{
    float a[3],b[3];
    a[0] = p[0] - radius;
    a[1] = p[1] - radius;
    a[2] = p[2] - radius;
    b[0] = p[0] + radius;
    b[1] = p[1] + radius;
    b[2] = p[2] + radius;
    drawBox(a,b);
}

//------------------------------------------------------------------------------------------
void drawColorAxes()
{
    glBegin(GL_LINES);
    glColor3f(1,0,0);
    glVertex3f(0,0,0);
    glVertex3f(1,0,0);
    glColor3f(0,1,0);
    glVertex3f(0,0,0);
    glVertex3f(0,1,0);
    glColor3f(0,0,1);
    glVertex3f(0,0,0);
    glVertex3f(0,0,1);
    glEnd();

    GLint ps;
    glGetIntegerv(GL_POINT_SIZE, &ps);
    glPointSize(5);
    glBegin(GL_POINTS);
    glColor3f(1,0,0);
    glVertex3f(1,0,0);
    glColor3f(0,1,0);
    glVertex3f(0,1,0);
    glColor3f(0,0,1);
    glVertex3f(0,0,1);
    glEnd();
    glPointSize(ps);
}

//------------------------------------------------------------------------------------------
void drawWirePlane(float r, int steps)
{
    float div = (r) / steps;
    vec3f first = make_vec3f(-r,0.f,-r);
    vec3f second = make_vec3f(r,0.f,r);

    glBegin(GL_LINES);

    for (float x=first.x; x<=second.x; x+=div)
    {
        //CT3(x,first.z,second.z);
        glVertex3f(x,0,first.z);
        glVertex3f(x,0,second.z);
    }

    for (float z=first.z; z<=second.z; z+=div)
    {
        glVertex3f(first.x,0,z);
        glVertex3f(second.x,0,z);
    }

    glEnd();
}

//------------------------------------------------------------------------------------------
bool __glCheckErrors(const char *file, const int line)
{
    bool first=true;

    for (;;)
    {
        int e=glGetError();

        if (e==GL_NO_ERROR)
        {
            if (!first)
                std::cerr << std::endl;
            return first;
        }

        first=false;

        static char s[1024];

        sprintf(s,"file <%s>, line %i:",file,line);

        if (e==GL_INVALID_ENUM)
            std::cerr<<"GL Error: "<<s<<" GL_INVALID_ENUM" << ", ";
        if (e==GL_INVALID_VALUE)
            std::cerr<<"GL Error: "<<s<<" GL_INVALID_VALUE" << ", ";
        if (e==GL_INVALID_OPERATION)
            std::cerr<<"GL Error: "<<s<<" GL_INVALID_OPERATION" << ", ";
        if (e==GL_STACK_OVERFLOW)
            std::cerr<<"GL Error: "<<s<<" GL_STACK_OVERFLOW" << ", ";
        if (e==GL_STACK_UNDERFLOW)
            std::cerr<<"GL Error: "<<s<<" GL_STACK_UNDERFLOW" << ", ";
        if (e==GL_OUT_OF_MEMORY)
            std::cerr<<"GL Error: "<<s<<" GL_OUT_OF_MEMORY" << ", ";
        else
            std::cerr << "GL Error: " << s << " "  << glewGetErrorString(e) << ", ";
    }
}

//------------------------------------------------------------------------------------------
#if defined(EASY_USE_GLUT) || 1
//static bool g_glut_initialized = false;
void drawText(int x, int y, const char *s, void *font)
{
    /*if (!g_glut_initialized)
    {
        int argc=1;
        char* argv[]={"glut_utils"};
        glutInit(&argc,argv);
        g_glut_initialized = true;
    }*/
    if (!font)
        font = GLUT_BITMAP_HELVETICA_10;
    glRasterPos2f(x, y);
    int len = (int)strlen(s);
    for (int i = 0; i < len; i++)
    {
        glutBitmapCharacter(font, s[i]);
    }
}
#else
void DrawText(int x, int y, const char *s, void *font)
{
	glRasterPos2f(x, y);
}
#endif

//------------------------------------------------------------------------------------------
/* draw sphere: (courtesy of http://local.wasp.uwa.edu.au/~pbourke)

   Create a sphere centered at c, with radius r, and precision n
   Draw a point for zero radius spheres
   Use CCW facet ordering
   "method" is 0 for quads, 1 for triangles
      (quads look nicer in wireframe mode)
   Partial spheres can be created using theta1->theta2, phi1->phi2
   in radians 0 < theta < 2pi, -pi/2 < phi < pi/2
*/
void drawSphere(float* c, float r, int n, int method, float theta1, float theta2, float phi1, float phi2)
{
    int i,j;
    float t1,t2,t3;
    float e[3],p[3];

    if (r < 0)
        r = -r;
    if (n < 0)
        n = -n;
    if (n < 4 || r <= 0)
    {
        glBegin(GL_POINTS);
        glVertex3f(c[0],c[1],c[2]);
        glEnd();
        return;
    }

    for (j=0; j<n/2; j++)
    {
        t1 = (phi1 + j * (phi2 - phi1) / (n/2));
        t2 = (phi1 + (j + 1) * (phi2 - phi1) / (n/2));

        if (method == 0)
            glBegin(GL_QUAD_STRIP);
        else
            glBegin(GL_TRIANGLE_STRIP);

        for (i=0; i<=n; i++)
        {
            t3 = (theta1 + i * (theta2 - theta1) / n);

            e[0] = cos(t1) * cos(t3);
            e[1] = sin(t1);
            e[2] = cos(t1) * sin(t3);
            p[0] = c[0] + r * e[0];
            p[1] = c[1] + r * e[1];
            p[2] = c[2] + r * e[2];
            glNormal3f(e[0],e[1],e[2]);
            glTexCoord2f(1-(i/(float)n), 2*j/(float)n);
            glVertex3f(p[0],p[1],p[2]);

            e[0] = cos(t2) * cos(t3);
            e[1] = sin(t2);
            e[2] = cos(t2) * sin(t3);
            p[0] = c[0] + r * e[0];
            p[1] = c[1] + r * e[1];
            p[2] = c[2] + r * e[2];
            glNormal3f(e[0],e[1],e[2]);
            glTexCoord2f(1-(i/(float)n), 2*(j+1)/(float)n);
            glVertex3f(p[0],p[1],p[2]);

        }
        glEnd();
    }
}

//------------------------------------------------------------------------------------------
void drawWireSphere(float* c, float r, int n, int method, float theta1, float theta2, float phi1, float phi2)
{
    glPushAttrib(GL_POLYGON_BIT);
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    drawSphere(c, r, n, method, theta1, theta2, phi1, phi2);
    glPopAttrib();
}

//-----------------------------------------------------------------------------------
void drawWireCylinder(float* a, float* b, float radiusA, float radiusB, int step, bool cap)
{
    if (a == b)
        return;

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();

    vec3f axis;
    axis.x = (b[0]-a[0]);
    axis.y = (b[1]-a[1]);
    axis.z = (b[2]-a[2]);
    // align to axis
    vec3f u = make_vec3f(0,0,1);

    vec3f unitAxis = axis.normalized();
    float dotau = dot(unitAxis, u);
    if (dotau < -0.5 || dotau > 0.5)
        u = make_vec3f(1,0,0);

    mat44f m = mat44f::fromVectors(normalize(cross(cross(unitAxis,u),unitAxis)), normalize(cross(unitAxis,u)), axis, vec3f::fromArray(a));
    glMultMatrixf((GLfloat*)&m);

    std::vector<vec3f> vp;

    for (float a=0; a<2.001; a+=1.f/step)
    {
        float an = a*PI;
        float ca = cosf(an);
        float sa = sinf(an);
        vec3f vca = make_vec3f(ca*radiusA, sa*radiusA, 0);
        vec3f vcb = make_vec3f(ca*radiusB, sa*radiusB, 1);

        vp.push_back(vca);
        vp.push_back(vcb);
    }

    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, &vp[0]);
    glDrawArrays(GL_LINES, 0, vp.size());
    glDisableClientState(GL_VERTEX_ARRAY);

    if(cap)
    {
        vec3f start = make_vec3f(0,0,0);
        vec3f end = make_vec3f(0,0,1);
        drawWireCircle((float*)&start, radiusA, std::max(16,step));
        drawWireCircle((float*)&end, radiusB, std::max(16,step));
    }

    glPopMatrix();
}

//-----------------------------------------------------------------------------------
void drawWireCircle(float* pos,float radius,float dstep,float from,float to)
{
    int step = std::max(int(fabsf(to-from)/dstep), 1);

    from = PI/180*from;
    to = PI/180*to;
    std::vector<vec3f> vp;
    for (int i=0; i<=step; ++i)
    {
        float an = (from+(to-from) * (float(i)/step));
        vp.push_back(vec3f::fromArray(pos) + radius * make_vec3f(cosf(an), sinf(an), 0));
    }

    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, &vp[0]);
    glDrawArrays(GL_LINE_STRIP, 0, vp.size());
    glDisableClientState(GL_VERTEX_ARRAY);
}

//------------------------------------------------------------------------------------------
}
}
