/* ---------------------------------------------------------------------------
 * This software is in the public domain, furnished "as is", without technical
 * support, and with no warranty, express or implied, as to its usefulness for
 * any purpose.

 * Author: Wil Braithwaite.
 *
 */

#ifndef MAT33F_H_INCLUDED
#define MAT33F_H_INCLUDED

#include "math_utils.h"

namespace Easy
{

//------------------------------------------------------------------------------------------
/// mat33f
/// multiplication is applied on left side first. so RotateZ * Scale = scale first, then rotate.
/// vectors are multiplied on left. i.e. RotateZ * v = rotated vector.
///
struct NVPARTICLES_ALIGN(16) mat33f
{
    union NVPARTICLES_ALIGN(16)
{
    vec4f m[3];
    float _array[12];
};

// access:
inline NVPARTICLES_CUDA_EXPORT float& element (const int row, const int col);
inline NVPARTICLES_CUDA_EXPORT float element (const int row, const int col) const;
inline NVPARTICLES_CUDA_EXPORT float& operator() (const int iRow, const int iCol);
inline NVPARTICLES_CUDA_EXPORT float operator() (const int iRow, const int iCol) const;

// constructions:
inline NVPARTICLES_CUDA_EXPORT static mat33f identity();
inline NVPARTICLES_CUDA_EXPORT static mat33f zero();
inline NVPARTICLES_CUDA_EXPORT static mat33f fromAxisAngle(const vec3f& axis, const float angle);
inline NVPARTICLES_CUDA_EXPORT static mat33f fromVectors(const vec4f& X, const vec4f& Y, const vec4f& Z);
inline NVPARTICLES_CUDA_EXPORT static mat33f fromVectors(const vec3f& X, const vec3f& Y, const vec3f& Z);
inline NVPARTICLES_CUDA_EXPORT static mat33f rotateX(const float rad);
inline NVPARTICLES_CUDA_EXPORT static mat33f rotateY(const float rad);
inline NVPARTICLES_CUDA_EXPORT static mat33f rotateZ(const float rad);
inline NVPARTICLES_CUDA_EXPORT static mat33f scale(const float sx, const float sy, const float sz);
//inline NVPARTICLES_CUDA_EXPORT static mat44f fromArray(float *s);
//inline NVPARTICLES_CUDA_EXPORT static mat44f rotateQ(vec4f quat);
inline NVPARTICLES_CUDA_EXPORT static mat33f fromOuter(float u1, float u2, float u3, float v1, float v2, float v3);

// operations:
inline NVPARTICLES_CUDA_EXPORT void toAxisAngle(vec3f& rkAxis, float& angle) const;
inline NVPARTICLES_CUDA_EXPORT float determinant() const;
inline NVPARTICLES_CUDA_EXPORT float trace() const;
inline NVPARTICLES_CUDA_EXPORT mat33f transposed() const;
inline NVPARTICLES_CUDA_EXPORT vec3f multiply(const vec3f& v) const;
inline NVPARTICLES_CUDA_EXPORT vec3f multiplyTranspose(const vec3f& v) const;
inline NVPARTICLES_CUDA_EXPORT vec4f multiply(const vec4f& v) const;
inline NVPARTICLES_CUDA_EXPORT vec4f multiplyTranspose(const vec4f& v) const;
inline NVPARTICLES_CUDA_EXPORT mat33f multiply(const mat33f &b) const;
inline NVPARTICLES_CUDA_EXPORT mat33f multiplyTranspose(const mat33f &tb) const;
inline NVPARTICLES_CUDA_EXPORT mat33f normalized() const;

// static operations:
inline NVPARTICLES_CUDA_EXPORT static mat33f transpose(const mat33f& m);
inline NVPARTICLES_CUDA_EXPORT static mat33f slerp(float t, const mat33f& a, const mat33f& b);

};

inline NVPARTICLES_CUDA_EXPORT mat33f operator*(const mat33f &a, const mat33f &b);
inline NVPARTICLES_CUDA_EXPORT vec3f operator*(const mat33f &m, const vec3f &v);
inline NVPARTICLES_CUDA_EXPORT vec4f operator*(const mat33f &m, const vec4f &v);

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT
float& mat33f::element (const int row, const int col)
{
    return _array[row | (col<<2)];
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT
float mat33f::element (const int row, const int col) const
{
    return _array[row | (col<<2)];
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT
float mat33f::operator() (const int row, const int col) const
{
    return element(row, col);
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT
float& mat33f::operator() (const int row, const int col)
{
    return element(row, col);
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT
mat33f mat33f::identity()
{
    mat33f m;
    m.m[0] = make_vec4f(1,0,0,0);
    m.m[1] = make_vec4f(0,1,0,0);
    m.m[2] = make_vec4f(0,0,1,0);
    return m;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT
mat33f mat33f::zero()
{
    mat33f m;
    m.m[0] = make_vec4f(0,0,0,0);
    m.m[1] = make_vec4f(0,0,0,0);
    m.m[2] = make_vec4f(0,0,0,0);
    return m;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT
mat33f mat33f::fromVectors(const vec4f& X, const vec4f& Y, const vec4f& Z)
{
    mat33f m;
    m.m[0] = make_vec4f(X.x,X.y,X.z,0);
    m.m[1] = make_vec4f(Y.x,Y.y,Y.z,0);
    m.m[2] = make_vec4f(Z.x,Z.y,Z.z,0);
    return m;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT
mat33f mat33f::fromVectors(const vec3f& X, const vec3f& Y, const vec3f& Z)
{
    mat33f m;
    m.m[0] = make_vec4f(X.x,X.y,X.z,0);
    m.m[1] = make_vec4f(Y.x,Y.y,Y.z,0);
    m.m[2] = make_vec4f(Z.x,Z.y,Z.z,0);
    return m;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f mat33f::rotateX(const float rad)
{
    mat33f m;
    m.m[0] = make_vec4f(1,0,0,0);
    m.m[1] = make_vec4f(0,cosf(rad),sinf(rad),0);
    m.m[2] = make_vec4f(0,-sinf(rad),cosf(rad),0);
    return m;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f mat33f::rotateY(const float rad)
{
    mat33f m;
    m.m[0] = make_vec4f(cosf(rad),0,-sinf(rad),0);
    m.m[1] = make_vec4f(0,1,0,0);
    m.m[2] = make_vec4f(sinf(rad),0,cosf(rad),0);
    return m;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f mat33f::rotateZ(const float rad)
{
    mat33f m;
    m.m[0] = make_vec4f(cosf(rad),sinf(rad),0,0);
    m.m[1] = make_vec4f(-sinf(rad),cosf(rad),0,0);
    m.m[2] = make_vec4f(0,0,1,0);
    return m;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f mat33f::scale(const float sx, const float sy, const float sz)
{
    mat33f m;
    m.m[0] = make_vec4f(sx,0,0,0);
    m.m[1] = make_vec4f(0,sy,0,0);
    m.m[2] = make_vec4f(0,0,sz,0);
    return m;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f mat33f::normalized() const
{
    mat33f r;
    r.m[0] = m[0].normalized();
    r.m[1] = m[1].normalized();
    r.m[2] = m[2].normalized();
    return r;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT void mat33f::toAxisAngle(vec3f& axis, float& angle) const
{
    // Let (x,y,z) be the unit-length axis and let A be an angle of rotation.
    // The rotation matrix is R = I + sin(A)*P + (1-cos(A))*P^2 where
    // I is the identity and
    //
    //       +-        -+
    //   P = |  0 -z +y |
    //       | +z  0 -x |
    //       | -y +x  0 |
    //       +-        -+
    //
    // If A > 0, R represents a counterclockwise rotation about the axis in
    // the sense of looking from the tip of the axis vector towards the
    // origin.  Some algebra will show that
    //
    //   cos(A) = (trace(R)-1)/2  and  R - R^t = 2*sin(A)*P
    //
    // In the event that A = pi, R-R^t = 0 which prevents us from extracting
    // the axis through P.  Instead note that R = I+2*P^2 when A = pi, so
    // P^2 = (R-I)/2.  The diagonal entries of P^2 are x^2-1, y^2-1, and
    // z^2-1.  We can solve these for axis (x,y,z).  Because the angle is pi,
    // it does not matter which sign you choose on the square roots.
    /*
    0 1 2
    3 4 5
    6 7 8

    0 3 6
    1 4 7
    2 5 8
    */


    float trace = element(0,0) + element(1,1) + element(2,2);
    angle = acosf((0.5f)*(trace-1.0f));  // in [0,PI]

    if ( angle > 0.0f )
    {
        if ( angle < PI )
        {
            axis.x = element(1,2) - element(2,1);
            axis.y = element(2,0) - element(0,2);
            axis.z = element(0,1) - element(1,0);
            axis.normalize();
        }
        else
        {
            // angle is PI
            float fHalfInverse;
            if ( element(0,0) >= element(1,1) )
            {
                if ( element(0,0) >= element(2,2) )
                {
                    axis.x = (0.5f)*sqrtf(element(0,0) - element(1,1) - element(2,2) + 1.0f);
                    fHalfInverse = (0.5f)/axis.x;
                    axis.y = fHalfInverse*element(1,0);
                    axis.z = fHalfInverse*element(2,0);
                    axis.normalize();
                }
                else
                {
                    axis.z = (0.5f)*sqrtf(element(2,2) - element(0,0) - element(1,1) + 1.0f);
                    fHalfInverse = (0.5f)/axis.z;
                    axis.x = fHalfInverse*element(2,0);
                    axis.y = fHalfInverse*element(2,1);
                    axis.normalize();
                }
            }
            else
            {
                if ( element(1,1) >= element(2,2) )
                {
                    axis.y = (0.5f)*sqrtf(element(1,1) - element(0,0) - element(2,2) + 1.0f);
                    fHalfInverse = (0.5f)/axis.y;
                    axis.x = fHalfInverse*element(1,0);
                    axis.z = fHalfInverse*element(2,1);
                    axis.normalize();
                }
                else
                {
                    axis.z = (0.5f)*sqrtf(element(2,2) - element(0,0) - element(1,1) + 1.0f);
                    fHalfInverse = (0.5f)/axis.z;
                    axis.x = fHalfInverse*element(2,0);
                    axis.y = fHalfInverse*element(2,1);
                    axis.normalize();
                }
            }
        }
    }
    else
    {
        // The angle is 0 and the matrix is the identity.  Any axis will
        // work, so just use the x-axis.
        axis.x = 1.0f;
        axis.y = 0.0f;
        axis.z = 0.0f;
    }
}

//----------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f mat33f::fromAxisAngle(const vec3f& rkAxis, const float angle)
{
    float fCos = cosf(angle);
    float fSin = sinf(angle);
    float fOneMinusCos = 1.0f-fCos;
    float fX2 = rkAxis.x*rkAxis.x;
    float fY2 = rkAxis.y*rkAxis.y;
    float fZ2 = rkAxis.z*rkAxis.z;
    float fXYM = rkAxis.x*rkAxis.y*fOneMinusCos;
    float fXZM = rkAxis.x*rkAxis.z*fOneMinusCos;
    float fYZM = rkAxis.y*rkAxis.z*fOneMinusCos;
    float fXSin = rkAxis.x*fSin;
    float fYSin = rkAxis.y*fSin;
    float fZSin = rkAxis.z*fSin;

    mat33f r = mat33f::zero();
    r.element(0,0) = fX2*fOneMinusCos+fCos;
    r.element(1,0) = fXYM-fZSin;
    r.element(2,0) = fXZM+fYSin;
    r.element(0,1) = fXYM+fZSin;
    r.element(1,1) = fY2*fOneMinusCos+fCos;
    r.element(2,1) = fYZM-fXSin;
    r.element(0,2) = fXZM-fYSin;
    r.element(1,2) = fYZM+fXSin;
    r.element(2,2) = fZ2*fOneMinusCos+fCos;

    return r;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT float mat33f::determinant() const
{
    float fCo00 = element(1,1)*element(2,2) - element(2,1)*element(1,2);
    float fCo10 = element(2,1)*element(0,2) - element(0,1)*element(2,2);
    float fCo20 = element(0,1)*element(1,2) - element(1,1)*element(0,2);
    float fDet = element(0,0)*fCo00 + element(1,0)*fCo10 + element(2,0)*fCo20;
    return fDet;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f mat33f::fromOuter(float u1, float u2, float u3, float v1, float v2, float v3)
{
    return fromVectors(
               make_vec3f(u1*v1, u1*v2, u1*v3),
               make_vec3f(u2*v1, u2*v2, u2*v3),
               make_vec3f(u3*v1, u3*v2 ,u3*v3)
           );
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT float mat33f::trace() const
{
    return m[0].x + m[1].y + m[2].z;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f mat33f::transposed() const
{
    mat33f r;
    r.element(0,0) = element(0,0);
    r.element(1,1) = element(1,1);
    r.element(2,2) = element(2,2);
    r.element(1,0) = element(0,1);
    r.element(2,0) = element(0,2);
    r.element(0,1) = element(1,0);
    r.element(0,2) = element(2,0);
    r.element(1,2) = element(2,1);
    r.element(2,1) = element(1,2);
    return r;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f mat33f::transpose(const mat33f& m)
{
    return m.transposed();
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT vec3f mat33f::multiply(const vec3f& v) const
{
    return transposed().multiplyTranspose(v);
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT vec4f mat33f::multiply(const vec4f& v) const
{
    return transposed().multiplyTranspose(v);
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT vec3f mat33f::multiplyTranspose(const vec3f& a) const
{
    vec3f r;
    //vec4f a = make_vec4f(v, 0.0f);
    r.x = a.x * m[0].x + a.y * m[0].y + a.z * m[0].z;// + a.w * m[0].w;//dot(v, m[0]);
    r.y = a.x * m[1].x + a.y * m[1].y + a.z * m[1].z;// + a.w * m[1].w;//dot(v, m[1]);
    r.z = a.x * m[2].x + a.y * m[2].y + a.z * m[2].z;// + a.w * m[2].w;//dot(v, m[2]);
    return r;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT vec4f mat33f::multiplyTranspose(const vec4f& a) const
{
    vec4f r;
    r.w = a.w;
    r.x = a.x * m[0].x + a.y * m[0].y + a.z * m[0].z;// + a.w * m[0].w;//dot(v, m[0]);
    r.y = a.x * m[1].x + a.y * m[1].y + a.z * m[1].z;// + a.w * m[1].w;//dot(v, m[1]);
    r.z = a.x * m[2].x + a.y * m[2].y + a.z * m[2].z;// + a.w * m[2].w;//dot(v, m[2]);
    return r;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f mat33f::multiply(const mat33f &b) const
{
    return multiplyTranspose(b.transposed());
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f mat33f::multiplyTranspose(const mat33f &tb) const
{
    mat33f r;
    for (int i=0; i<3; ++i)
    {
        r.m[i].x = m[i].x * tb.m[0].x + m[i].y * tb.m[0].y + m[i].z * tb.m[0].z;//r.m[i].x = dot(m[i], tb.m[0]);
        r.m[i].y = m[i].x * tb.m[1].x + m[i].y * tb.m[1].y + m[i].z * tb.m[1].z;//r.m[i].y = dot(m[i], tb.m[1]);
        r.m[i].z = m[i].x * tb.m[2].x + m[i].y * tb.m[2].y + m[i].z * tb.m[2].z;//r.m[i].z = dot(m[i], tb.m[2]);
    }
    return r;
}

//------------------------------------------------------------------------------------------
/// Eigen decomposition code
/// thanks to Matthias Mueller-Fischer!
///
inline NVPARTICLES_CUDA_EXPORT void jacobiRotate(mat33f& A, mat33f& R, int p, int q)
{
    // rotates A through phi in pq-plane to set A(p,q) = 0
    // rotation stored in R whose columns are eigenvectors of A
    if (A(p,q) == 0.0f)
        return;

    float d = (A(p,p) - A(q,q))/(2.0f*A(p,q));
    float t = 1.0f / (fabs(d) + sqrtf(d*d + 1.0f));
    if (d < 0.0f) t = -t;
    float c = 1.0f/sqrtf(t*t + 1.0f);
    float s = t*c;
    A(p,p) += t*A(p,q);
    A(q,q) -= t*A(p,q);
    A(p,q) = A(q,p) = 0.0f;
    // transform A
    int k;
    for (k = 0; k < 3; k++)
    {
        if (k != p && k != q)
        {
            float Akp = c*A(k,p) + s*A(k,q);
            float Akq =-s*A(k,p) + c*A(k,q);
            A(k,p) = A(p,k) = Akp;
            A(k,q) = A(q,k) = Akq;
        }
    }
    // store rotation in R
    for (k = 0; k < 3; k++)
    {
        float Rkp = c*R(k,p) + s*R(k,q);
        float Rkq =-s*R(k,p) + c*R(k,q);
        R(k,p) = Rkp;
        R(k,q) = Rkq;
    }
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT void eigenDecomposition(mat33f& A, mat33f& R)
{
    const int numJacobiIterations = 4;
    const float epsilon = 1e-15f;

    // only for symmetric matrices!
    R = mat33f::identity();
    int iter = 0;
    while (iter < numJacobiIterations)  	// 3 off diagonal elements
    {
        // find off diagonal element with maximum modulus
        int p,q;
        float a,max;
        max = fabs(A(0,1));
        p = 0;
        q = 1;
        a   = fabs(A(0,2));
        if (a > max)
        {
            p = 0;
            q = 2;
            max = a;
        }
        a   = fabs(A(1,2));
        if (a > max)
        {
            p = 1;
            q = 2;
            max = a;
        }
        // all small enough -> done
        if (max < epsilon)
            break;
        // rotate matrix with respect to that element
        jacobiRotate(A, R, p,q);
        iter++;
    }
}

//------------------------------------------------------------------------------------------
/// operators:
//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f operator* (const mat33f& a, const float b)
{
    return mat33f::fromVectors(a.m[0]*b, a.m[1]*b, a.m[2]*b);
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f operator* (const float b, const mat33f& a)
{
    return mat33f::fromVectors(a.m[0]*b, a.m[1]*b, a.m[2]*b);
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f operator/ (const mat33f& a, const float b)
{
    return mat33f::fromVectors(a.m[0]/b, a.m[1]/b, a.m[2]/b);
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f operator/ (const float b, const mat33f& a)
{
    return mat33f::fromVectors(b/a.m[0], b/a.m[1], b/a.m[2]);
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f operator+(const mat33f& a, const mat33f& b)
{
    return mat33f::fromVectors(a.m[0]+b.m[0], a.m[1]+b.m[1], a.m[2]+b.m[2]);
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f operator-(const mat33f& a, const mat33f& b)
{
    return mat33f::fromVectors(a.m[0]-b.m[0], a.m[1]-b.m[1], a.m[2]-b.m[2]);
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT void operator*=(mat33f& a, const float b)
{
    a.m[0] *= b;
    a.m[1] *= b;
    a.m[2] *= b;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT void operator+=(mat33f& a, const mat33f& b)
{
    a.m[0] += b.m[0];
    a.m[1] += b.m[1];
    a.m[2] += b.m[2];
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT bool operator==(const mat33f& a, const mat33f& b)
{
    return (a.m[0] == b.m[0] && a.m[1] == b.m[1] && a.m[2] == b.m[2]);
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT bool operator!=(const mat33f& a, const mat33f& b)
{
    return (a.m[0] != b.m[0] || a.m[1] != b.m[1] || a.m[2] == b.m[2]);
}

//------------------------------------------------------------------------------------------
///
//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f mat33f::slerp(float t, const mat33f& a, const mat33f& b)
{
    // remove scaling.
    mat33f an = a.normalized();
    // transform b into a-space.
    mat33f delta = b.normalized() * an.transposed();

    // interpolate the original scale...
    float asx = length(make_vec3f(a.m[0]));
    float asy = length(make_vec3f(a.m[1]));
    float asz = length(make_vec3f(a.m[2]));
    float bsx = length(make_vec3f(b.m[0]));
    float bsy = length(make_vec3f(b.m[1]));
    float bsz = length(make_vec3f(b.m[2]));
    float sx = t*bsx + (1-t)*asx;
    float sy = t*bsy + (1-t)*asy;
    float sz = t*bsz + (1-t)*asz;

    // interpolate the angle...
    vec3f axis;
    float angle;
    delta.toAxisAngle(axis, angle);

    // reconstruct delta-matrix.
    delta = fromAxisAngle(axis, t*angle);

    return scale(sx, sy, sz) * delta * an;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT mat33f operator*(const mat33f &a, const mat33f &b)
{
    return a.multiply(b);
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT vec4f operator*(const mat33f &m, const vec4f &v)
{
    return m.multiply(v);
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT vec3f operator*(const mat33f &m, const vec3f &v)
{
    return m.multiply(v);
}

//------------------------------------------------------------------------------------------
#ifndef __CUDA_ARCH__
inline std::ostream &operator<< (std::ostream &s, const mat33f &m)
{
    for(int r=0; r<3; ++r)
    {
        s << "[ ";
        for(int c=0; c<3; ++c)
        {
            s << m(r,c) << " ";
        }
        s << "] ";
    }
    return s;
}
#endif
//------------------------------------------------------------------------------------------
}

#endif // MAT33F_H_INCLUDED
