/* ---------------------------------------------------------------------------
 * This software is in the public domain, furnished "as is", without technical
 * support, and with no warranty, express or implied, as to its usefulness for
 * any purpose.

 * Author: Wil Braithwaite.
 *
 */

#include "std_utils.h"
#include <sstream>
#include <sys/types.h>
#include <signal.h>
#include <errno.h>

using namespace std;

namespace Easy
{

//-----------------------------------------------------------------------------------
Stringf::Stringf(const std::string& format, ...)
{
    char temp[10000];
    va_list args;
    va_start(args, format);
    vsprintf(temp, format.c_str(), args);
    va_end(args);
    ((std::string *)this)->operator=(temp);
}

//-----------------------------------------------------------------------------------
Stringf::Stringf(const char *format,...)
{
    char temp[10000];
    va_list args;
    va_start(args, format);
    vsprintf(temp, format, args);
    va_end(args);
    ((std::string *)this)->operator=(temp);
}

//-----------------------------------------------------------------------------------
StringArray::StringArray()
{
}

//-----------------------------------------------------------------------------------
StringArray::StringArray(const char* s, const char* seperators)
{
    set(std::string(s), std::string(seperators));
}

//-----------------------------------------------------------------------------------
StringArray::StringArray(const std::string& s, const std::string& seperators)
{
    set(s, seperators);
}

//-----------------------------------------------------------------------------------
bool StringArray::contains(const std::string& s)
{
    for (const_iterator it=begin(); it!=end(); ++it)
        if (*it == s)
            return true;
    return false;
}

//-----------------------------------------------------------------------------------
void StringArray::set(const std::string& s, const std::string& seperators)
{
    clear();
    std::string remaining = s;
    const char *sep = seperators.c_str();

    int n = strspn(remaining.c_str(), sep);
    if (n != 0)
        remaining = remaining.substr(n, remaining.length()-n);

    for (;;)
    {
        int n = strcspn(remaining.c_str(), sep);
        if (n==0)
            break;

        push_back(remaining.substr(0, n));

        remaining = remaining.substr(n, remaining.length()-n);

        n = strspn(remaining.c_str(), sep);
        if (n == 0)
            break;

        remaining = remaining.substr(n, remaining.length()-n);
    }
}

//-----------------------------------------------------------------------------------
bool isFile(const std::string& path)
{
    if (path.empty())
        return false;

#if defined( _WIN32 )
    return (GetFileAttributesA(path.c_str()) != INVALID_FILE_ATTRIBUTES);
#else
    return (access(path.c_str(), F_OK|R_OK) == 0);
#endif
}

//-----------------------------------------------------------------------------------
ScopedTimer::TimerMap ScopedTimer::global_timers;

int ScopedTimer::g_indent=0;

//-----------------------------------------------------------------------------------
ScopedTimer::ScopedTimer(const char *m,int *indent_storage, bool use_global, float s)
        :
        indent(indent_storage),
        use_global_timers(use_global),
        scale(s)
{
    if (!indent)
        indent = &g_indent;
    ++(*indent);
    strcpy(message,m);
    timer.Reset();
}

//-----------------------------------------------------------------------------------
ScopedTimer::~ScopedTimer()
{
    float v = timer.Value();
    --(*indent);

    if (use_global_timers)
    {
        TimerMap::iterator it = global_timers.find(message);
        if (it == global_timers.end())
            global_timers[message] = std::make_pair(0.f,0);
        else
        {
            it->second.first += v;
            it->second.second ++;
        }
    }
    else
    {
        for (int i=(*indent);i>0;--i)
            fprintf(stdout,"\t");//fprintf(stderr,"\t");
        fprintf(stdout,"%08.6fs: %s\n",v*scale,(const char *)message);
        //fprintf(stderr,"%08.6fs: %s\n",timer.Value(),(const char *)message);
    }
}

//-----------------------------------------------------------------------------------
void ScopedTimer::Print(const char *s)
{
    for (int i=(*indent)-1; i>0; --i)
        fprintf(stdout, "\t");
    fprintf(stdout,"%s\n", s);
}

//-----------------------------------------------------------------------------------
void ScopedTimer::ResetGlobalTimers()
{
    global_timers.clear();
}

//-----------------------------------------------------------------------------------
void ScopedTimer::DumpGlobalTimers(float scale)
{
    TimerMap::iterator it;

    size_t len = 3;
    for (it = global_timers.begin();it != global_timers.end(); it++)
        len += (*it).first.length() + 3;

    for(size_t c=len; c>0; --c)
        std::cout << "-";
    std::cout << std::endl;

    std::cout << " | ";
    for (it = global_timers.begin();it != global_timers.end(); it++)
    {
        std::cout << (*it).first << " | ";
    }
    std::cout << std::endl << "   ";

    //std::stringstream ss(stringstream::in);
    for (it = global_timers.begin();it != global_timers.end(); it++)
    {
        size_t len = ((*it).first.length()+3);
        std::stringstream ss(stringstream::out);
        ss << (it->second.first / max(1,it->second.second))*scale;
        string s = ss.str();
        std::cout << s;

        len -= s.length();
        for(size_t c=len; c>0; --c)
            std::cout << " ";
    }
    std::cout << std::endl;

    for(size_t c=len; c>0; --c)
        std::cout << "-";
    std::cout << std::endl;
}

//-----------------------------------------------------------------------------------
float ScopedTimer::GlobalTimerValue(const char *s)
{
    if (global_timers.find(s) != global_timers.end())
    {
        return ScopedTimer::global_timers[s].first / ScopedTimer::global_timers[s].second;
    }
    return -1;
}

//-----------------------------------------------------------------------------------
#if defined(_WIN32)
//unsigned __stdcall thread_func(void *t)
DWORD WINAPI thread_func( LPVOID t )
#else
void *thread_func(void *t)
#endif
{
    Thread *task = (Thread *)t;

#if defined(_WIN32)
    //prctl(PR_TERMCHILD);
#else
    pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS,NULL);
#endif

    task->_isComplete = false;
    task->onExecute();
    task->_isComplete = true;

    task->_returnCode = 0;

#if defined(_WIN32)
    //_endthreadex(0);
#else
    try
    {
        pthread_exit(&task->_returnCode);
    }
    catch(...)
    {
        throw;
    }
#endif

    return (0);
}

//-----------------------------------------------------------------------------------
Thread::Thread()
{
    _thread = 0;
}

//-----------------------------------------------------------------------------------
Thread::~Thread()
{
    killWait();
#if defined(_WIN32)
    CloseHandle(_thread);
#else
#endif
}

//-----------------------------------------------------------------------------------
bool Thread::execute()
{
#if defined(_WIN32)
    //thread = (HANDLE)beginthreadex(NULL, 0, (unsigned (*)(void *))thread_func, this, 0, NULL);
    _thread = CreateThread(NULL, 0, thread_func, this, 0, NULL);
    if (_thread == 0)
        return false;
#else
    int r = pthread_create(&_thread, NULL, thread_func, this);
    if (r != 0)
    {
        perror("Couldn't create thread!");
        return false;
    }

#endif
    return true;
}

//-----------------------------------------------------------------------------------
void Thread::wait()
{
    if (!_thread)
        return;
#if defined(_WIN32)
    WaitForSingleObject(_thread, INFINITE );
    //CloseHandle(thread);
#else
    int rc=pthread_join(_thread,NULL);
    if (rc!=0)
    {
        printf("pthread error code: %d\n",rc);
    }

    _thread = 0;
#endif
}

//-----------------------------------------------------------------------------------
void Thread::kill()
{
    if (!_thread)
        return;
#if defined(_WIN32)
    TerminateThread(_thread, 0);
#else

    if (_isComplete == false)
    {
        pthread_cancel(_thread);
    }
#endif
}

//-----------------------------------------------------------------------------------
void Thread::killWait()
{
    kill();
    wait();
}

//-----------------------------------------------------------------------------------
bool Thread::isCompleted()
{
#if defined(_WIN32)
    return(_isComplete);//(waitpid(pid,NULL,WNOHANG)!=0);
#else
    return(_isComplete);
#endif
}

}
