/* ---------------------------------------------------------------------------
 * This software is in the public domain, furnished "as is", without technical
 * support, and with no warranty, express or implied, as to its usefulness for
 * any purpose.

 * Author: Wil Braithwaite.
 *
 */

#ifndef STD_UTILS_H_INCLUDED
#define STD_UTILS_H_INCLUDED

#include <string>
#include <iostream>
#include <cstring>
#include <map>
#include <vector>
#include <cstdarg>
#include <cstdlib>
#include <cstdio>
#include <cassert>

#define strcasecmp stricmp

#if defined(_WIN32)

#ifndef NOMINMAX
#define NOMINMAX
#endif
#ifndef _WIN32_WINNT
#define _WIN32_WINNT 0x0502
#endif
#define WIN32_LEAN_AND_MEAN
#include <windows.h>

#else

#include <pthread.h>
#include <errno.h>
#include <semaphore.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/time.h>

#endif

#if defined(_WIN32)
#if !defined(_WIN32_WINNT)
// The following Windows API function is declared explicitly;
// otherwise any user would have to specify /D_WIN32_WINNT=0x0400
extern "C" BOOL WINAPI TryEnterCriticalSection( LPCRITICAL_SECTION );
#endif
#endif


#ifndef STDERR_DEFINED
#define STDERR_DEFINED
#define STDERRSEP()      std::cerr << "---------------------------" << std::endl
#define STDERR(a)        std::cerr << #a ": "<<(a)<<std::endl
#define STDERR2(a,b)     std::cerr << #a ": "<<(a)<<"    "<< #b ": "<<(b)<<std::endl
#define STDERR3(a,b,c)   std::cerr << #a ": "<<(a)<<"    "<< #b ": "<<(b)<<"    "<< #c ": "<<(c)<<std::endl
#define STDERR4(a,b,c,d) std::cerr << #a ": "<<(a)<<"    "<< #b ": "<<(b)<<"    "<< #c ": "<<(c)<<"    "<< #d ": "<<(d)<<std::endl
#define STDERR5(a,b,c,d,e) std::cerr << #a ": "<<(a)<<"    "<< #b ": "<<(b)<<"    "<< #c ": "<<(c)<<"    "<< #d ": "<<(d)<<"    "<< #e ": "<<(e)<<std::endl
#define STDL() std::cerr << std::string(__FILE__) << ":" << int(__LINE__) << std::endl
#define STDERRDUMP(a) a.Dump(#a)
#endif

namespace Easy
{


//-----------------------------------------------------------------------------------
/// Generic singleton class.
///
template <typename T>
class Singleton
{
private:
    Singleton(const Singleton<T> &);
    Singleton& operator=(const Singleton<T> &);

protected:
    static T* _singleton;

public:
    Singleton( void )
    {
        assert( !_singleton );
#if defined( _MSC_VER ) && _MSC_VER < 1200
        int offset = (int)(T*)1 - (int)(Singleton <T>*)(T*)1;
        _singleton = (T*)((int)this + offset);
#else
        _singleton = static_cast< T* >( this );
#endif
    }

    ~Singleton( void )
    {
        assert( _singleton );
        _singleton = 0;
    }

    static T& getSingleton( void )
    {
        assert(_singleton);
        return (*_singleton);
    }
};

//-----------------------------------------------------------------------------------
class StringArray : public std::vector<std::string>
{
public:
    StringArray();
    StringArray(const char* s, const char* seperators=",");
    //StringArray(const StringArray &sl);
    StringArray(const std::string &text, const std::string &seperators=",");

    void set(const std::string &text, const std::string &seperators=",");

    bool contains(const std::string& text);
};

//-----------------------------------------------------------------------------------
bool isFile(const std::string& path);

//-----------------------------------------------------------------------------------
class Timer
{
public:
    double msInvFrequency;

#if defined(_WIN32)
    mutable double mLastQueryTime;

    Timer()
    {
		msInvFrequency = -1.0;
        Reset();
    }

    void Reset()
    {
		if (msInvFrequency<=0)
		{
			LARGE_INTEGER lFreq;
			QueryPerformanceFrequency(&lFreq);
			msInvFrequency = 1.0 / double(lFreq.QuadPart);
		}
		LARGE_INTEGER lCount;
		QueryPerformanceCounter(&lCount);
		mLastQueryTime = (lCount.QuadPart * msInvFrequency);
    }

    float Value()
    {
		double lastTime = mLastQueryTime;
		LARGE_INTEGER lCount;
		QueryPerformanceCounter(&lCount);
		return (float)((lCount.QuadPart * msInvFrequency) - lastTime);
    }

#else
    long start_sec;
    long start_usec;

    Timer()
    {
        Reset();
    }

    void Reset()
    {
        struct timeval tv;
        struct timezone tz;
        gettimeofday(&tv,&tz);
        start_sec=tv.tv_sec;
        start_usec=tv.tv_usec;
    }

    float Value()
    {
        struct timeval tv;
        struct timezone tz;
        gettimeofday(&tv,&tz);
        return((tv.tv_sec-start_sec)+(tv.tv_usec-start_usec)/1000000.0);
    }
#endif
};

/// class to construct printf-formating with std::string
class Stringf : public std::string
{
public:
    Stringf(const char *format, ...);
    Stringf(const std::string& format, ...);
};


class ScopedTimer
{

    static int g_indent;
    Timer timer;

    char message[256];
    int *indent;
    bool use_global_timers;
    float scale;
    typedef std::map<std::string, std::pair<float,int> > TimerMap;

public:
    ScopedTimer(const char *m="",int *indent_storage=NULL, bool use_global=false, float scale=1.f);

    ~ScopedTimer();

    void Print(const char *s);

    static void ResetGlobalTimers();
    static void DumpGlobalTimers(float scale=1);

    static float GlobalTimerValue(const char *s);

    static TimerMap global_timers;
};

#define EASY_TIMED(func) \
    {\
    ScopedTimer td( #func , 0, true);\
    func;\
    }

/// return a random number between 0 and 1.
inline float frand()
{
    return rand() / (float) RAND_MAX;
}

/// return a random number between -1 and +1.
inline float sfrand()
{
    return (rand() / (float) RAND_MAX)*2.0f-1.0f;
}

#define FORCE_MAX(var,lo) var=((var)<(lo))?(lo):(var)
#define FORCE_MIN(var,hi) var=((var)>(hi))?(hi):(var)
#define FORCE_MINMAX(var,lo,hi) var=(((var)<(lo))?(lo):((var)>(hi))?(hi):(var))
#define FORCE_TOGGLE(var) (var=!(var))

//----------------------------------------------------------------------------------------------
#if !defined(_WIN32)

inline int __nsleep(const struct timespec *req, struct timespec *rem)
{
    struct timespec temp_rem;
    if(nanosleep(req,rem)==-1)
        return __nsleep(rem, &temp_rem);
    else
        return 1;
}
//----------------------------------------------------------------------------------------------
inline int msleep(unsigned long msec)
{
    struct timespec req= {0,0},rem= {0,0};
    time_t sec=(int)(msec/1000);
    msec=msec-(sec*1000);
    req.tv_sec=sec;
    req.tv_nsec=msec*1000000L;
    __nsleep(&req,&rem);
    return 1;
}

#endif

//----------------------------------------------------------------------------------------------
class Thread
{
    friend void *func(void *t);

#if defined(_WIN32)
    HANDLE _thread;
#else
    pthread_t _thread;
#endif

public:
    int _returnCode;
    bool _isComplete;

public:
    Thread();
    virtual ~Thread();

    /// Create and run.
    bool execute();

    /// Execute the work of the task.
    virtual void onExecute() = 0;

public:
    /// Only sends the message.
    void kill();

    /// Sends the message and then waits.
    void killWait();

    /// Waits until it has finished.
    void wait();

    /// Returns whether it has completed without interruption.
    bool isCompleted();
};

//----------------------------------------------------------------------------------------------
class Mutex
{
private:
#if defined(_WIN32)
    typedef CRITICAL_SECTION MUTEX_TYPE;
#else
    typedef pthread_mutex_t MUTEX_TYPE;
#endif
    MUTEX_TYPE mutex;

public:
    Mutex()
    {
#if defined(_WIN32)
        //mutex = CreateMutex(NULL,FALSE,NULL);
        InitializeCriticalSection(&mutex);
#else
        pthread_mutex_init(&mutex,NULL);
#endif
    }

    ~Mutex()
    {
#if defined(_WIN32)
        //CloseHandle(mutex);
        DeleteCriticalSection(&mutex);
#else
        pthread_mutex_destroy(&mutex);
#endif
    }

    void claim()
    {
#if defined(_WIN32)
        //DWORD dwWaitResult = WaitForSingleObject(mutex,INFINITE);
        EnterCriticalSection(&mutex);
#else
        pthread_mutex_lock(&mutex);
#endif
    }

    bool tryClaim()
    {
#if defined(_WIN32)
        //return(WaitForSingleObject(mutex,0)==WAIT_OBJECT_0);
        return (TryEnterCriticalSection(&mutex)!=0);
#else
        return(pthread_mutex_trylock(&mutex)==0);
#endif
    }

    void release()
    {
#if defined(_WIN32)
        //ReleaseMutex(mutex);
        LeaveCriticalSection(&mutex);
#else
        pthread_mutex_unlock(&mutex);
#endif
    }

    bool isLocked()
    {
#if defined(_WIN32)
        //return(WaitForSingleObject(mutex,0)==WAIT_TIMEOUT);
        return (TryEnterCriticalSection(&mutex)==0);
#else
        return(pthread_mutex_trylock(&mutex)==EBUSY);
#endif
    }

    MUTEX_TYPE &asNative()
    {
        return mutex;
    }
};

//----------------------------------------------------------------------------------------------
class ScopedMutex
{
    Mutex& _mutex;
public:
    ScopedMutex(Mutex& mutex)
        :
        _mutex(mutex)
    {
        _mutex.claim();
    }

    ~ScopedMutex()
    {
        _mutex.release();
    }
};

//----------------------------------------------------------------------------------------------
#if defined(_WIN32)
#else
void Semaphore_Init();
#define ustestsema(x) (x.__align)
#endif

class Semaphore
{
private:
#if defined(_WIN32)
    unsigned int volatile count;
    HANDLE sema;
#else
    sem_t sema;
#endif

public:
    Semaphore(int _count=1)
    {
#if defined(_WIN32)
        count = (_count);
        sema = CreateSemaphore(
                   NULL,           // default security attributes
                   count,  // initial count
                   100,  // maximum count
                   NULL);          // unnamed semaphore
#else
        int rc=sem_init(&sema,0,_count);
        assert(rc==0);
#endif
    }

    ~Semaphore()
    {
#if defined(_WIN32)
        CloseHandle(sema);
#else
        int rc=sem_destroy(&sema);
        assert(rc==0);
#endif
    }

    inline void wait()
    {
        claim();
    }

    inline void signal()
    {
        release();
    }

    void claim()
    {
        //fprintf(stderr,"Claiming %p %d\n",this,ustestsema(sema));
#if defined(_WIN32)
        if (!tryClaim(INFINITE))
            perror("Semaphore: wait failed");
#else
        int rc=0;
        do
        {
            rc = sem_wait(&sema);
        }
        while(rc==-1 && errno==EINTR);

        if(rc != 0)
            perror("sem_wait failed");
#endif
        //fprintf(stderr,"Claimed %p %d\n",this,ustestsema(sema));
    }

    bool tryClaim(unsigned int timeoutms=0L)
    {
        //fprintf(stderr,"Claiming %p %d\n",this,ustestsema(sema));
#if defined(_WIN32)
        DWORD dwMilliseconds = DWORD(timeoutms);
        switch (WaitForSingleObject(sema, dwMilliseconds))
        {
        case WAIT_OBJECT_0:
            InterlockedDecrement((LONG*)&count);
            return true;

        case WAIT_TIMEOUT:
            return false;

        default:
            perror("Semaphore: WaitForSingleObject failed");
            return false;
        }
        //return true;
#else
        //return(sem_trywait(&sema)==0);
        if (timeoutms == 0)
            return (sem_trywait(&sema) == 0);

        Timer timer;
        timeoutms += (unsigned int)(timer.Value()*1000.f);
        do
        {
            int rc = sem_trywait(&sema);
            if (rc == 0)
                return true;
            msleep(1);
        }
        while ( (unsigned int)(timer.Value()*1000.f) < timeoutms );
        return false;
#endif
        //fprintf(stderr,"Claimed %p %d\n",this,ustestsema(sema));
    }

    void release()
    {
        //fprintf(stderr,"Releasing %p %d\n",this,ustestsema(sema));
#if defined(_WIN32)
        InterlockedIncrement((LONG*)&count);
        if ( ReleaseSemaphore(sema, 1, NULL) == FALSE )
        {
            InterlockedDecrement((LONG*)&count);
            perror("Semaphore: post failed");
        }

#else
        int rc=sem_post(&sema);
        assert(rc==0);
#endif
        //fprintf(stderr,"Released %p %d\n",this,ustestsema(sema));
    }

    int value()
    {
#if defined(_WIN32)
        return count;
#else
        int v;
        sem_getvalue(&sema,&v);
        return(v);
#endif
    }
};

//----------------------------------------------------------------------------------------------
template<class T>
class SharedPtr
{
protected:
    T* _pointer;
    unsigned int* _useCount;
    mutable Mutex* _lock;

public:

    /** Constructor.
    You must call bind before using object.
    */
    SharedPtr()
        :
        _pointer(0),
        _useCount(0),
        _lock(0)
    {
    }

    /** Constructor.
    @param ptr The pointer to take ownership of
    */
    template< class Y>
    SharedPtr(Y* ptr)
        :
        _pointer(ptr),
        _useCount(ptr ? new unsigned int : 0),
        _lock(0)
    {
        if (ptr)
        {
            *_useCount = 1;
            _lock = new Mutex();
        }
    }
    SharedPtr(const SharedPtr& r)
        :
        _pointer(0),
        _useCount(0),
        _lock(0)
    {
        if (r._lock)
        {
            ScopedMutex l(*r._lock);
            _lock = r._lock;
            _pointer = r._pointer;
            _useCount = r._useCount;
            // Handle zero pointer gracefully to manage STL containers
            if (_useCount)
                ++(*_useCount);
        }
    }

    SharedPtr& operator=(const SharedPtr& r)
    {
        if (_pointer == r._pointer)
            return *this;
        // Swap current data into a local copy
        // this ensures we deal with rhs and this being dependent
        SharedPtr<T> tmp(r);
        swap(tmp);
        return *this;
    }

    template< class Y>
    SharedPtr(const SharedPtr<Y>& r)
        :
        _pointer(0),
        _useCount(0),
        _lock(0)
    {
        if (r._lock)
        {
            ScopedMutex l(*r._lock);
            _lock = r._lock;
            /// is this crazy to cast this? I need to support dynamic casts i.e.
            /// derivedptr = static_cast<derivedptr>(baseptr);
            _pointer = (T*)r.pointer();
            _useCount = r.useCountPointer();
            // Handle zero pointer gracefully to manage STL containers
            if (_useCount)
                ++(*_useCount);
        }
    }

    template< class Y>
    SharedPtr& operator=(const SharedPtr<Y>& r)
    {
        if (_pointer == r.pointer())
            return *this;
        // Swap current data into a local copy
        // this ensures we deal with rhs and this being dependent
        SharedPtr<T> tmp(r);
        swap(tmp);
        return *this;
    }

    virtual ~SharedPtr()
    {
        _release();
    }

    inline operator T*() const
    {
        return _pointer;
    }

    inline T& operator*() const
    {
        assert(_pointer);
        return *_pointer;
    }

    inline T* operator->() const
    {
        assert(_pointer);
        return _pointer;
    }

    inline T* pointer() const
    {
        return _pointer;
    }

    void bind(T* ptr)
    {
        assert(!_pointer && !_useCount);
        _lock = new Mutex();
        ScopedMutex l(*_lock);
        _useCount = new unsigned int;
        _pointer = ptr;
    }

    inline bool unique() const
    {
        ScopedMutex l(*_lock);
        assert(_useCount);
        bool rc = (*_useCount == 1);
        return rc;
    }

    inline unsigned int useCount() const
    {
        ScopedMutex l(*_lock);
        assert(_useCount);
        unsigned int rc = (*_useCount);
        return rc;
    }

    inline unsigned int* useCountPointer() const
    {
        return _useCount;
    }

    inline bool valid() const
    {
        return (_pointer != NULL);
    }

    inline void setNull(void)
    {
        if (_pointer)
        {
            // can't scope lock mutex before release in case deleted
            _release();
            _pointer = 0;
            _useCount = 0;
        }
    }

protected:

    inline void _release(void)
    {
        bool destroyThis = false;

        if (_lock)
        {
            ScopedMutex l(*_lock);
            if (_useCount)
            {
                if (--(*_useCount) == 0)
                    destroyThis = true;
            }
        }

        if (destroyThis)
            _destroy();

        _lock = 0;
    }

    virtual void _destroy(void)
    {
        delete _pointer;
        delete _useCount;
        delete _lock;
    }

public:
    virtual void swap(SharedPtr<T> &other)
    {
        std::swap(_pointer, other._pointer);
        std::swap(_useCount, other._useCount);
        std::swap(_lock, other._lock);
    }
};

template<class T, class U> inline bool operator==(SharedPtr<T> const& a, SharedPtr<U> const& b)
{
    return a.pointer() == b.pointer();
}

template<class T, class U> inline bool operator!=(SharedPtr<T> const& a, SharedPtr<U> const& b)
{
    return a.pointer() != b.pointer();
}

template<class T, class U> inline bool operator<(SharedPtr<T> const& a, SharedPtr<U> const& b)
{
    return std::less<const void*>()(a.pointer(), b.pointer());
}

//----------------------------------------------------------------------------------------------
/// Generic factory class.
///
template <class T>
class GenericFactory
{
public:
    typedef T *(*CreatorFunc)();

    GenericFactory() {}

    const std::vector<std::string>& getTypes() const
    {
        return keys;
    }

    bool registerType(const std::string& type, CreatorFunc func)
    {
        int foundIndex = -1;
        for (int i=0; i<keys.size(); ++i)
            if (keys[i] == type)
            {
                foundIndex = i;
                break;
            }

        if (foundIndex == -1)
        {
            keys.push_back(type);
            creatorFuncs.push_back(func);
			return true;
        }
		return false;
    }

    bool deregisterType(const std::string& type)
    {
		int foundIndex = -1;
        for (int i=0; i<keys.size(); ++i)
            if (keys[i] == type)
            {
                foundIndex = i;
                break;
            }

        if (foundIndex >= 0)
        {
            keys.erase(keys.begin()+foundIndex);
            creatorFuncs.erase(creatorFuncs.begin()+foundIndex);
			return true;
        }
		return false;
    }

    T* create(const std::string& type) const
    {
        // try each of the specs and see if we have a match
        int foundIndex = -1;
        for (int i=0; i<keys.size(); ++i)
            if (keys[i] == type)
            {
                foundIndex = i;
                break;
            }

        if (foundIndex < 0)
            return 0;

        T* obj = creatorFuncs[foundIndex]();
        return obj;
    }

private:

    std::vector<std::string> keys;
    std::vector<CreatorFunc> creatorFuncs;

    GenericFactory(const GenericFactory &); // not implemented
    GenericFactory &operator=(const GenericFactory &); // not implemented
};

//----------------------------------------------------------------------------------------------
}

#endif // STD_UTILS_H_INCLUDED
