/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#include "gl_utils.h"

#include "NvParticlesParticleShape.h"
#include "NvParticlesPrimitiveShape.h"
#include "NvParticlesMayaStringResources.h"

#include <maya/MStatus.h>
#include <maya/MGlobal.h>
#include <maya/MObject.h>
#include <maya/MTypeId.h>

#include <maya/MAnimControl.h>
#include <maya/MSelectionList.h>

#include <maya/MItDependencyGraph.h>

#include <maya/MDataBlock.h>
#include <maya/MDataHandle.h>
#include <maya/MPlug.h>
#include <maya/MTime.h>
#include <maya/MVector.h>
#include <maya/MDoubleArray.h>
#include <maya/MFloatArray.h>
#include <maya/MIntArray.h>
#include <maya/MVectorArray.h>
#include <maya/MPlugArray.h>
#include <maya/MPointArray.h>
#include <maya/MFloatPointArray.h>
#include <maya/MFloatVector.h>
#include <maya/MDagPath.h>
#include <maya/MMatrix.h>
#include <maya/MFloatMatrix.h>

#include <maya/MFnTypedAttribute.h>
#include <maya/MFnNumericAttribute.h>
#include <maya/MFnCompoundAttribute.h>
#include <maya/MFnMessageAttribute.h>
#include <maya/MFnEnumAttribute.h>
#include <maya/MFnUnitAttribute.h>
#include <maya/MFnMatrixAttribute.h>
#include <maya/MFnGenericAttribute.h>

#include <maya/MFnDoubleArrayData.h>
#include <maya/MFnVectorArrayData.h>
#include <maya/MFnArrayAttrsData.h>
#include <maya/MFnStringData.h>
#include <maya/MFnStringArrayData.h>
#include <maya/MFnMatrixData.h>

#include <maya/MFnCamera.h>
#include <maya/MFnMesh.h>
#include <maya/MFnMeshData.h>

#include <maya/MFnField.h>
#include <maya/MFnGravityField.h>
#include <maya/MFnDragField.h>
#include <maya/MFnRadialField.h>
#include <maya/MFnVortexField.h>
#include <maya/MFnTurbulenceField.h>
#include <maya/MFnUniformField.h>
#include <maya/MFnVolumeAxisField.h>
#include <maya/MFnNewtonField.h>
#include <maya/MFnAirField.h>
#include <maya/MDynamicsUtil.h>

#include <maya/MMutexLock.h>
#include <maya/MItMeshPolygon.h>

#include <cassert>
#include "maya_utils.h"
#include "NvParticlesProfiler.h"
#include "NvParticlesManager.h"
#include "NvParticlesParticleRenderer.h"
#include "NvParticlesParticleModifier.h"
#include "NvParticlesParticleSolverImpl.h"
#include "NvParticlesParticleContainer.h"
#include "NvParticlesPrimitives.h"
#include "NvParticlesForces.h"

#include <algorithm>

#define NVPARTICLESFORMAYA_USE_ASYNC_UPDATE

/*
 * Known bugs:
 *
 * I acknowledge I've not been particularly clever with this implementation.
 * It's a demo :-)
 *
 * Due to primitives requiring persistence between frames for slerping transforms,
 * if you rename a primitive in maya, then it will fail to remove it from the
 * solver's primitives list when it is deleted in maya.
 * This can be solved by simply storing a "key" inside the primitive node, and then
 * using that as the solver's primitive name, rather than the dagnode's "partialName".
 * Field connections to solver forces due not suffer from this. They are
 * cleared every iteration as they do not require persistence between updates.
*/

using namespace Easy;
using std::cerr;
using std::cout;

static void drawMesh(TriangleMesh* triMesh);

//-----------------------------------------------------------------------------------
//
//-----------------------------------------------------------------------------------
MCallbackId NvParticlesParticleShape::nodeRemovedCallback;
MCallbackId NvParticlesParticleShape::nodePreConnectionCallback;
MTypeId NvParticlesParticleShape::id(kNvParticlesParticleShapePluginId);

// viewport 2.0
MString NvParticlesParticleShape::drawDbClassification("drawdb/geometry/" kNvParticlesParticleShapePluginName);
MString NvParticlesParticleShape::drawRegistrantId("NvParticlesParticleShapePlugin");

MObject NvParticlesParticleShape::attrDebugLevel;

MObject NvParticlesParticleShape::attrDummy;
MObject NvParticlesParticleShape::attrEmitters;
MObject NvParticlesParticleShape::attrFields;
MObject NvParticlesParticleShape::attrPrimitives;
MObject NvParticlesParticleShape::attrIsFull;
MObject NvParticlesParticleShape::attrStartTime;
MObject NvParticlesParticleShape::attrInheritFactor;
MObject NvParticlesParticleShape::attrTimeStep;
MObject NvParticlesParticleShape::attrSeed;
MObject NvParticlesParticleShape::attrEnable;

MObject NvParticlesParticleShape::attrRendererType;

// rendering attributes... (until we move into same mechanism as solver!)
MObject NvParticlesParticleShape::attrRendererRadius;
MObject NvParticlesParticleShape::attrRendererRadiusFactor;
MObject NvParticlesParticleShape::attrRendererPointSize;
MObject NvParticlesParticleShape::attrRendererColor;
MObject NvParticlesParticleShape::attrRendererUseLighting;
MObject NvParticlesParticleShape::attrRendererStreakWidth;
MObject NvParticlesParticleShape::attrRendererStreakLength;
MObject NvParticlesParticleShape::attrRendererUseColor;
MObject NvParticlesParticleShape::attrRendererDepthSort;

MObject NvParticlesParticleShape::attrDrawGrid;
MObject NvParticlesParticleShape::attrDrawPrimitives;
MObject NvParticlesParticleShape::attrDrawBounds;
MObject NvParticlesParticleShape::attrDrawDebugTextures;

MObject NvParticlesParticleShape::cudaDeviceAttr;
MObject NvParticlesParticleShape::attrOutArrayAttrs;
MObject NvParticlesParticleShape::attrMaxParticles;
MObject NvParticlesParticleShape::attrNumParticles;
MObject NvParticlesParticleShape::attrLastTime;
MObject NvParticlesParticleShape::attrDeltaTime;
MObject NvParticlesParticleShape::attrCurrentTime;
MObject NvParticlesParticleShape::attrBoundsLow;
MObject NvParticlesParticleShape::attrBoundsHigh;
MObject NvParticlesParticleShape::attrSolverEnable;

MObject NvParticlesParticleShape::attrGlTexDisplacementSize;
MObject NvParticlesParticleShape::attrInGeometry;
MObject NvParticlesParticleShape::attrInGeometryMat;
MObject NvParticlesParticleShape::attrInCameraInvMat;

// these attribute objects are created from the parameter definitions...
std::vector< std::pair<MObject, Easy::NvParticles::ParameterSpecPtr> > NvParticlesParticleShape::_solverAttributes;

//-----------------------------------------------------------------------------------
#define STRINGIFY(A) #A
#define STRINGIFY2(A) STRINGIFY(A)

//------------------------------------------------------------------------------------------
// ocean height-field shaders

static const char *vertexShader = STRINGIFY(

uniform mat4 modelToWorldMat;

void main()
{
    vec4 pos = vec4(gl_Vertex.xyz, 1.0);
    vec4 worldPos = modelToWorldMat * pos;
    gl_Position = gl_ModelViewProjectionMatrix * pos;
    gl_Normal = gl_NormalMatrix * gl_Normal;
	gl_FrontColor = vec4(gl_Normal.xyz,1);
    gl_TexCoord[0] = vec4(worldPos.xyz, 1);
}
);

static const char *fragmentShader = STRINGIFY(

void main()
{
    vec3 worldPos = gl_TexCoord[0].xyz;
    gl_FragColor = vec4(worldPos.xyz, gl_Color.a);
}
);




//-----------------------------------------------------------------------------------
NvParticlesParticleShape::NvParticlesParticleShape()
	:
	_nvParticles(0),
	particleRenderer(0),
	particleContainer(0),
    particleSolver(0)
{
    _nvParticles = &NvParticles::Manager::getSingleton();

    _timeIsConnected = false;
    _dummyIsConnected = false;
	_isConstructed = false;

    particleContainer = new Easy::NvParticles::ParticleContainer;
    particleSolver = new Easy::NvParticles::ParticleSolver;
    particleSolver->setDebugLevel(1);
    particleSolver->setContainer(particleContainer);

    _fbo = 0;
    _program = 0;
    _pboId = 0;
    _glTexId = 0;
    _glTexDepthId = 0;
    _triMesh = 0;
}
//-----------------------------------------------------------------------------------
NvParticlesParticleShape::~NvParticlesParticleShape()
{
    _destroy();
}

//-----------------------------------------------------------------------------------
bool NvParticlesParticleShape::isTransparent() const
{
    return true;
}

//-----------------------------------------------------------------------------------
bool NvParticlesParticleShape::drawLast() const
{
    /// CAVEAT:
    // This doesn't work when we use high quality mode.
    return true;
}

//-----------------------------------------------------------------------------------
bool NvParticlesParticleShape::isBounded() const
{
    /// CAVEAT:
    // If we return true, then the renderer culls the shape.
    // But we affect the shearXY plug and it modifies the bbox.
	/// TODO:
	// Find a better way to force the redraw.
    return false;
}

//-----------------------------------------------------------------------------------
void NvParticlesParticleShape::postConstructor()
{
    MFnDependencyNode nodeFn(thisMObject());
    nodeFn.setName("nvParticlesParticleShape#");
	_isConstructed = true;
}

//-----------------------------------------------------------------------------------
/// Return the bounding-box.
/// The box is calculated inside the compute method for efficiency.
/// If the particles are empty, then we return a null bbox: (0,0,0)(0,0,0)
///
MBoundingBox NvParticlesParticleShape::boundingBox() const
{
    float3 low = {0,0,0};
    float3 high = {0,0,0};

    if (particleContainer)
    {
        particleContainer->getBounds(low, high);
    }

    return MBoundingBox( MPoint(low[0], low[1], low[2]), MPoint(high[0], high[1], high[2]) );
}

//-----------------------------------------------------------------------------------
MStatus NvParticlesParticleShape::initialize()
{
    MStatus status = MS::kSuccess;
    MFnNumericAttribute nAttrFn;
    MFnTypedAttribute tAttrFn;
    MFnCompoundAttribute cAttrFn;
    MFnMessageAttribute mAttrFn;
    MFnEnumAttribute eAttrFn;
    MFnUnitAttribute uAttrFn;
	MFnGenericAttribute gAttrFn;
    MFnMatrixAttribute xAttrFn;

    MFnStringData stringDataFn;
	MFnStringArrayData	stringArrayDataFn;
	MStatus status2;

    // dummy output:
    attrDummy = nAttrFn.create( "dummy", "d", MFnNumericData::kFloat, 0, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setKeyable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setHidden(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setConnectable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrDummy ));

    attrEnable = nAttrFn.create( "enable", "ena", MFnNumericData::kBoolean, true, &status );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrEnable ) );

    attrDebugLevel = nAttrFn.create( "debugLevel", "dbg", MFnNumericData::kInt, 0, &status );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMin(0) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMax(2) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrDebugLevel ) );

    attrMaxParticles = nAttrFn.create( "maxCount", "mxc", MFnNumericData::kInt, 500000, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));
	CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setConnectable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMin(256) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMax(50000000) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrMaxParticles));

    attrBoundsLow = nAttrFn.create( "bboxLow", "bbl", MFnNumericData::k3Float, 0, &status );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setKeyable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setHidden(false) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrBoundsLow ));

    attrBoundsHigh = nAttrFn.create( "bboxHigh", "bbh", MFnNumericData::k3Float, 0, &status );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setKeyable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setHidden(false) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrBoundsHigh ));

    attrEmitters = tAttrFn.create( "emitters", "emts", MFnData::kDynArrayAttrs, &status);
    CHECK_MSTATUS_AND_RETURN_IT( tAttrFn.setReadable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( tAttrFn.setArray(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrEmitters));

    attrFields = mAttrFn.create( "fields", "flds", &status);
    CHECK_MSTATUS_AND_RETURN_IT( mAttrFn.setReadable(false));
    CHECK_MSTATUS_AND_RETURN_IT( mAttrFn.setArray(true));
    mAttrFn.setHidden(false);
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrFields));

    attrPrimitives = mAttrFn.create( "primitives", "prms", &status);
    CHECK_MSTATUS_AND_RETURN_IT( mAttrFn.setReadable(false));
    CHECK_MSTATUS_AND_RETURN_IT( mAttrFn.setArray(true));
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrPrimitives));

	// drawing diagnostics:

    attrDrawPrimitives = nAttrFn.create( "drawPrimitives", "drp", MFnNumericData::kBoolean, false, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrDrawPrimitives));

    attrDrawBounds = nAttrFn.create( "drawBounds", "drb", MFnNumericData::kBoolean, true, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrDrawBounds));

    attrDrawGrid = nAttrFn.create( "drawGrid", "drg", MFnNumericData::kBoolean, false, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrDrawGrid));

    attrDrawDebugTextures = nAttrFn.create( "drawDebugTextures", "drdt", MFnNumericData::kFloat, 0, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMin(0) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMax(1) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrDrawDebugTextures));

    // solver attributes:

    NvParticles::ParticleSolverSpecPtr spec = NvParticles::Manager::getSingleton().getParticleSolverSpec("wcsph");
    if (spec)
    {
        NvParticles::ParameterSpecs::const_iterator it = spec->parameterDefinitions.begin();
        for (; it != spec->parameterDefinitions.end(); ++it)
        {
            std::string name = "solver_" + it->first;

            NvParticles::ParameterSpecPtr spec = it->second;
            MObject attr = MObject::kNullObj;

            if (spec->getType() == NvParticles::ParameterSpec::FLOAT)
            {
                attr = nAttrFn.create( name.c_str(), name.c_str(), MFnNumericData::kDouble, spec->_defaultValue.asFloat(), &status);
                CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
                CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));

                if (spec->getMeta().contains("ui-min"))
                    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMin(spec->getMeta().asFloat("ui-min")));
                if (spec->getMeta().contains("ui-max"))
                    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMax(spec->getMeta().asFloat("ui-max")));
                if (spec->getMeta().contains("ui-softmin"))
                    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setSoftMin(spec->getMeta().asFloat("ui-softmin")));
                if (spec->getMeta().contains("ui-softmax"))
                    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setSoftMax(spec->getMeta().asFloat("ui-softmax")));
            }
            else if (spec->getType() == NvParticles::ParameterSpec::INT)
            {
                std::string enums = spec->getMeta().asString("enum");
                if (enums.length() > 0)
                {
                    StringArray sa(enums);
                    attr = eAttrFn.create( name.c_str(), name.c_str(), 0, &status );
                    CHECK_MSTATUS_AND_RETURN_IT( eAttrFn.setKeyable(false) );
                    CHECK_MSTATUS_AND_RETURN_IT( eAttrFn.setConnectable(false) );
                    for (int i=0; i<sa.size(); ++i)
                    {
                        CHECK_MSTATUS_AND_RETURN_IT( eAttrFn.addField(sa[i].c_str(), i) );
                    }
                }
                else
                {
                    attr = nAttrFn.create( name.c_str(), name.c_str(), MFnNumericData::kInt, spec->_defaultValue.asInt(), &status);
                    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
                    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));

                    if (spec->getMeta().contains("ui-min"))
                        CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMin(spec->getMeta().asInt("ui-min")));
                    if (spec->getMeta().contains("ui-max"))
                        CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMax(spec->getMeta().asInt("ui-max")));
                    if (spec->getMeta().contains("ui-softmin"))
                        CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setSoftMin(spec->getMeta().asInt("ui-softmin")));
                    if (spec->getMeta().contains("ui-softmax"))
                        CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setSoftMax(spec->getMeta().asInt("ui-softmax")));
                }
            }
            else if (spec->getType() == NvParticles::ParameterSpec::BOOL)
            {
                attr = nAttrFn.create( name.c_str(), name.c_str(), MFnNumericData::kBoolean, spec->_defaultValue.asBool(), &status);
                CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
                CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));
            }
            else if (spec->getType() == NvParticles::ParameterSpec::MATRIX)
            {
                attr = xAttrFn.create( name.c_str(), name.c_str(), MFnMatrixAttribute::kFloat, &status );
                mat44f m = it->second->_defaultValue.asMatrix44();
                float m16[4][4];
                memcpy(m16, &m, sizeof(float)*4*4);
                xAttrFn.setDefault(MFloatMatrix(m16));
                CHECK_MSTATUS_AND_RETURN_IT( xAttrFn.setWritable(true));
                CHECK_MSTATUS_AND_RETURN_IT( xAttrFn.setReadable(true));
            }
            else if (spec->getType() == NvParticles::ParameterSpec::STRING)
            {
                attr = tAttrFn.create(name.c_str(), name.c_str(), MFnData::kString, stringDataFn.create(spec->_defaultValue.asString().c_str(), &status2), &status);
            	CHECK_MSTATUS_AND_RETURN_IT( status2 );
	            CHECK_MSTATUS_AND_RETURN_IT( status );
                CHECK_MSTATUS_AND_RETURN_IT( tAttrFn.setWritable(true));
                CHECK_MSTATUS_AND_RETURN_IT( tAttrFn.setReadable(true));
            }

            if (attr != MObject::kNullObj)
            {
                CHECK_MSTATUS_AND_RETURN_IT( addAttribute(attr));

                _solverAttributes.push_back(std::make_pair(attr, spec));
            }
        }
    }

    attrGlTexDisplacementSize = nAttrFn.create( "oceanTextureSize", "oceanTexSize", MFnNumericData::kInt, 0, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMin(0));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMax(1024));

    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrGlTexDisplacementSize));

    attrInGeometry = gAttrFn.create("inOceanGeometry", "inOceanGeom");
    CHECK_MSTATUS_AND_RETURN_IT( gAttrFn.addAccept(MFnData::kMesh) );
    CHECK_MSTATUS_AND_RETURN_IT( gAttrFn.addAccept(MFnData::kDynSweptGeometry) );
    CHECK_MSTATUS_AND_RETURN_IT( gAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( gAttrFn.setWritable(true) );
	CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrInGeometry ) );

    attrInGeometryMat = xAttrFn.create("inOceanGeometryMat", "inOceanGeomMat");
	CHECK_MSTATUS_AND_RETURN_IT( xAttrFn.setHidden(false) );
    CHECK_MSTATUS_AND_RETURN_IT( xAttrFn.setReadable(false));
    CHECK_MSTATUS_AND_RETURN_IT( xAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( xAttrFn.setWritable(true) );
	CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrInGeometryMat ) );

    attrInCameraInvMat = xAttrFn.create("inOceanCameraInvMat", "inOceanCamInvMat");
	CHECK_MSTATUS_AND_RETURN_IT( xAttrFn.setHidden(false) );
    CHECK_MSTATUS_AND_RETURN_IT( xAttrFn.setReadable(false));
    CHECK_MSTATUS_AND_RETURN_IT( xAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( xAttrFn.setWritable(true) );
	CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrInCameraInvMat ) );

    // misc:

    attrNumParticles = nAttrFn.create( "count", "cnt", MFnNumericData::kInt, 0, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(false));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(false));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setHidden(true));
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrNumParticles));

    // startTime:
    attrStartTime = nAttrFn.create( "startTime", "stt", MFnNumericData::kDouble, 1, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrStartTime));

    // timeStep:
    attrTimeStep = nAttrFn.create( "timeStep", "ts", MFnNumericData::kDouble, 0.04, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setKeyable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMin(0.001) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMax(1.0) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrTimeStep));

    // deltaTime: (required for emitter to compute)
    attrDeltaTime = nAttrFn.create( "deltaTime", "odt", MFnNumericData::kDouble, 1.0, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setHidden(false));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setKeyable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrDeltaTime));

    // inheritFactor: (required for emitter to compute)
    attrInheritFactor = nAttrFn.create( "inheritFactor", "ihn_", MFnNumericData::kDouble, 1.0, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrInheritFactor));

    // isFull: (tell emitter not to produce more particles)
    attrIsFull = nAttrFn.create( "isFull", "ifl", MFnNumericData::kBoolean, false, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setHidden(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrIsFull));

    // seed:
    attrSeed = nAttrFn.create( "seed", "sd", MFnNumericData::kInt, 0, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setKeyable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrSeed));

    // cuda device:
    cudaDeviceAttr = nAttrFn.create( "cudaDevice", "cud", MFnNumericData::kInt, 0, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setReadable(true));
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setKeyable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( cudaDeviceAttr));

    // time:
    attrCurrentTime = uAttrFn.create( "currentTime", "cti", MFnUnitAttribute::kTime, 0.0 );
    CHECK_MSTATUS_AND_RETURN_IT( uAttrFn.setReadable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( uAttrFn.setWritable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( uAttrFn.setHidden(true) );
    CHECK_MSTATUS_AND_RETURN_IT( uAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( uAttrFn.setKeyable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( uAttrFn.setConnectable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrCurrentTime ) );

	// output the buffer arrays:
	MFnArrayAttrsData fnArrayAttrsData;
    MObject objArrayAttrsData = fnArrayAttrsData.create ( &status );
	CHECK_MSTATUS_AND_RETURN_IT(status);
    attrOutArrayAttrs = tAttrFn.create( "outArrays", "oarr", MFnData::kDynArrayAttrs, objArrayAttrsData, &status );
    CHECK_MSTATUS_AND_RETURN_IT( tAttrFn.setWritable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( tAttrFn.setStorable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrOutArrayAttrs) );

    // output lastTime:
    attrLastTime = nAttrFn.create( "lastTime", "oldt", MFnNumericData::kDouble, 0.0, &status);
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setWritable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setHidden(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrLastTime) );

    //------------------------------------------------------------------------
    // solver attributes:

    attrSolverEnable = nAttrFn.create( "enableSolver", "e_so", MFnNumericData::kBoolean, true, &status );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrSolverEnable ) );

    //------------------------------------------------------------------------
    // renderer attributes:

    attrRendererType = tAttrFn.create( "renderer", "ren",  MFnData::kString, stringDataFn.create("points"), &status );
    CHECK_MSTATUS_AND_RETURN_IT( tAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( tAttrFn.setKeyable(false) );
    CHECK_MSTATUS_AND_RETURN_IT( tAttrFn.setConnectable(false) );

    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrRendererType ) );

    attrRendererRadius = nAttrFn.create( "renderer_radius", "rrad", MFnNumericData::kFloat, 0, &status );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMin(0.0) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setSoftMax(10.0) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setKeyable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setConnectable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrRendererRadius ) );

    attrRendererRadiusFactor = nAttrFn.create( "renderer_radiusFactor", "rraf", MFnNumericData::kFloat, 1, &status );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setSoftMin(0.001) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setSoftMax(5.0) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setKeyable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setConnectable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrRendererRadiusFactor ) );

    attrRendererPointSize = nAttrFn.create( "renderer_pointSize", "rps", MFnNumericData::kInt, 2, &status );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMin(1) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setMax(75) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setKeyable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setConnectable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrRendererPointSize ) );

    attrRendererColor = nAttrFn.createColor( "renderer_color", "rcol" );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrRendererColor ) );

    attrRendererUseLighting = nAttrFn.create( "renderer_useLighting", "rul", MFnNumericData::kBoolean, true, &status );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setKeyable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setConnectable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrRendererUseLighting ) );

    attrRendererUseColor = nAttrFn.create( "renderer_useColor", "ruc", MFnNumericData::kBoolean, false, &status );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setKeyable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setConnectable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrRendererUseColor ) );

    attrRendererStreakLength = nAttrFn.create( "renderer_streakLength", "rstl", MFnNumericData::kFloat, 0, &status );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setSoftMin(0) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setSoftMax(5) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setKeyable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setConnectable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrRendererStreakLength ) );

    attrRendererStreakWidth = nAttrFn.create( "renderer_streakWidth", "rstw", MFnNumericData::kFloat, 2, &status );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setStorable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setSoftMin(1) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setSoftMax(5) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setKeyable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( nAttrFn.setConnectable(true) );
    CHECK_MSTATUS_AND_RETURN_IT( addAttribute( attrRendererStreakWidth ) );

    return status;
}

//-----------------------------------------------------------------------------------
MStatus NvParticlesParticleShape::setDependentsDirty(MPlug const & inPlug, MPlugArray& affectedPlugs)
{
    if ( inPlug.attribute() == attrCurrentTime || inPlug.attribute() == attrMaxParticles )
    {
        MObject thisNode = thisMObject();

        MPlug plug(thisNode, attrDummy);
        affectedPlugs.append(plug);

        // can we move these to be dependent on dummy?
        //
        plug.setAttribute(attrOutArrayAttrs);
        affectedPlugs.append(plug);

        return MS::kSuccess;
    }
    else if ( inPlug.attribute() == attrDummy )
    {
        MObject thisNode = thisMObject();

        MPlug plug(thisNode, attrOutArrayAttrs);
        affectedPlugs.append(plug);
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------------
MStatus NvParticlesParticleShape::_updateParticlesFromAttributes(MDataBlock& block)
{
    MStatus status = MS::kUnknownParameter;
    MDataHandle inputData;

    NvParticles::Parameters attributes;

    inputData = block.inputValue ( attrLastTime, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);
    float lastTime = inputData.asDouble();

    inputData = block.inputValue ( attrCurrentTime, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);
    float currentTime = inputData.asTime().as(MTime::uiUnit());
    attributes["time"].setFloat(currentTime);

    inputData = block.inputValue ( attrTimeStep, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);
    float timeStep = inputData.asDouble();
    attributes["frameRate"].setFloat(1.0f/timeStep);

    inputData = block.inputValue ( attrStartTime, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);
    float startTime = inputData.asDouble();

    inputData = block.inputValue ( attrMaxParticles, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);
    int maxParticles = inputData.asInt();

    inputData = block.inputValue ( attrSeed, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);
    attributes["seed"].setInt(inputData.asInt());

	inputData = block.inputValue ( attrInheritFactor, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);
    attributes["inheritFactor"].setFloat(inputData.asDouble());

    // solver attributes:

    for (int i=0; i<_solverAttributes.size(); ++i)
    {
        MObject attr = _solverAttributes[i].first;
        NvParticles::ParameterSpecPtr spec = _solverAttributes[i].second;

        inputData = block.inputValue ( attr, &status );
        CHECK_MSTATUS_AND_RETURN_IT(status);

        if (spec->getType() == NvParticles::ParameterSpec::FLOAT)
        {
            attributes[spec->getName()].setFloat(inputData.asDouble());
        }
        else if (spec->getType() == NvParticles::ParameterSpec::INT)
        {
            attributes[spec->getName()].setInt(inputData.asInt());
        }
        else if (spec->getType() == NvParticles::ParameterSpec::STRING)
        {
            attributes[spec->getName()].setString(inputData.asString().asChar());
        }
        else if (spec->getType() == NvParticles::ParameterSpec::BOOL)
        {
            attributes[spec->getName()].setBool(inputData.asBool());
        }
        else if (spec->getType() == NvParticles::ParameterSpec::MATRIX)
        {
            MFloatMatrix fm = inputData.asFloatMatrix();
            attributes[spec->getName()].setMatrix44(mat44f::fromArray((float*)&fm));
        }
    }

    // use the internal values.
    inputData = block.inputValue ( attrGlTexDisplacementSize, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);
    attributes["glTexDisplacementSize"].setInt(inputData.asInt());
    attributes["glTexDisplacement"].setInt(_pboId);

	inputData = block.inputValue ( attrDebugLevel, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);
    int debugLevel = inputData.asInt();
	particleSolver->setDebugLevel(debugLevel);

    inputData = block.inputValue ( cudaDeviceAttr, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);
    int cudaDevice = inputData.asInt();

	/// CAVEAT:
	// if we are going to change then we must destroy the particles
	// BEFORE we create new CUDA contexts.
	// Alternatively, we could remove the Sync and Destroy from Setup.
    if (cudaDevice != particleSolver->_cudaDevice || particleContainer->maxParticles != maxParticles)
	{
        // wipe the solver (and all the GPU resources) before we change the configuration...
        particleSolver->destroy();
	    
        particleContainer->setMaxParticles(maxParticles);
        particleContainer->addBuffer(NvParticles::ParticleBufferSpec("position", NvParticles::ParticleBufferSpec::FLOAT4));
        particleContainer->addBuffer(NvParticles::ParticleBufferSpec("velocity", NvParticles::ParticleBufferSpec::FLOAT4));
        particleContainer->addBuffer(NvParticles::ParticleBufferSpec("color", NvParticles::ParticleBufferSpec::FLOAT4));
        particleContainer->addBuffer(NvParticles::ParticleBufferSpec("id", NvParticles::ParticleBufferSpec::UINT));
        particleContainer->addBuffer(NvParticles::ParticleBufferSpec("birthTime", NvParticles::ParticleBufferSpec::FLOAT));

		// setup the cuda context we are using.
#ifdef _WIN32
		particleSolver->setCudaDeviceConfig(cudaDevice, M3dView::active3dView().deviceContext(), M3dView::active3dView().display());
#else
		particleSolver->setCudaDeviceConfig(cudaDevice, M3dView::active3dView().display(), M3dView::active3dView().glxContext());
#endif
	}

    particleSolver->setSolverType("wcsph");
    particleSolver->setParameters(attributes);
    particleSolver->updateContext();

    particleContainer->setExportedBuffers("");

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------------
MStatus NvParticlesParticleShape::compute( const MPlug& plug, MDataBlock& block )
{
    MStatus status = MS::kUnknownParameter;
    MDataHandle inputData;

    if (!_nvParticles)
        return MS::kFailure;

    if (!particleContainer)
        return MS::kFailure;

    if( plug == attrDummy )
    {
        gl::initialize();

        // the time has been updated, so do an update...

        inputData = block.inputValue ( attrEnable, &status );
        CHECK_MSTATUS_AND_RETURN_IT(status);
        bool enabled = inputData.asBool();
        if (!enabled)
        {
            CHECK_MSTATUS_AND_RETURN_IT( block.setClean( plug ) );
            return MS::kSuccess;
        }

        inputData = block.inputValue ( attrLastTime, &status );
        CHECK_MSTATUS_AND_RETURN_IT(status);
        float lastFrame = inputData.asDouble();

        inputData = block.inputValue ( attrCurrentTime, &status );
        CHECK_MSTATUS_AND_RETURN_IT(status);
        float currentFrame = inputData.asTime().as(MTime::uiUnit());
        float currentTime = inputData.asTime().as(MTime::kSeconds);

        inputData = block.inputValue ( attrStartTime, &status );
        CHECK_MSTATUS_AND_RETURN_IT(status);
        float startFrame = inputData.asDouble();

        particleSolver->sync();

        _updateOcean(block);

        _updateParticlesFromAttributes(block);


        if (currentFrame == startFrame)
        {
            usedMessageList.clear();
            NvParticles::Profiler::getSingleton().Reset();

            if (particleSolver->_context)
            {
                // clear these for good measure.
                particleSolver->_context->particleForces->clear();
                particleSolver->_context->primitives->clear();
            }
            particleSolver->setTime(currentFrame);

            particleContainer->reset();

            status = _readEmitters(plug, block);
            status = _readFields();
            status = _readPrimitives();

            // force the update on this start frame.
			particleSolver->updateAsync(false, true);
            // we need another one because the render buffers are double buffered.
			particleSolver->updateAsync(false, true);
            particleSolver->sync();
        }
        else //if (currentFrame >= lastFrame)
        {
            /// CAVEAT: do we need to force only evaluating when going forwards in time?

            float frameDiff = currentFrame - lastFrame;

            particleSolver->setTime(currentFrame);

            status = _readEmitters( plug, block );
            status = _readFields();
            status = _readPrimitives();

            if (Easy::getBoolAttr(block, attrSolverEnable))
            {
                particleSolver->updateAsync();

#if !defined(NVPARTICLESFORMAYA_USE_ASYNC_UPDATE)
                particleSolver->sync();
#endif
            }
        }

        lastFrame = currentFrame;
        MDataHandle hLastTime = block.outputValue( attrLastTime, &status );
        CHECK_MSTATUS_AND_RETURN_IT(status);
        hLastTime.setDouble(lastFrame);
        hLastTime.setClean();

        // update the particle count
        MDataHandle hNumParticles = block.outputValue( attrNumParticles, &status );
        CHECK_MSTATUS_AND_RETURN_IT(status);
        hNumParticles.setInt(particleContainer->getParticleCount());
        hNumParticles.setClean();


        MDataHandle hDummy = block.outputValue( attrDummy, &status );
        CHECK_MSTATUS_AND_RETURN_IT(status);
        hDummy.setFloat(lastFrame);
        hDummy.setClean();
        //CHECK_MSTATUS_AND_RETURN_IT( block.setClean( plug ) );

        NvParticles::Profiler::getSingleton().IncrementFrame();

        return MS::kSuccess;
    }
    else if(plug == attrOutArrayAttrs)
    {
        // ensure an update at this frame.
        inputData = block.inputValue ( attrDummy, &status );
        CHECK_MSTATUS_AND_RETURN_IT(status);
        float t = inputData.asFloat();

        _computeOutputArrayAttrs(plug, block);

        CHECK_MSTATUS_AND_RETURN_IT( block.setClean( plug ) );
        return MS::kSuccess;
    }


    return status;
}

//-----------------------------------------------------------------------------------
MStatus NvParticlesParticleShape::_updateOcean(MDataBlock& block)
{
    MDataHandle inData, outData;
    MStatus status;

    if (_program == 0)
    {
        _program = new GLSLProgram(vertexShader, fragmentShader);
    }

    MDataHandle geomData = block.inputValue ( attrInGeometry, &status );
    MObject geomObj = geomData.data();
    _updateMeshData(geomObj);

    if (_triMesh == 0)
    {
        return MS::kSuccess;
    }

    float m16f[4][4];
    inData = block.inputValue ( attrInGeometryMat, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);
    inData.asMatrix().get(m16f);
    mat44f geomXform = mat44f::fromArray((float*)m16f);

    mat44f cameraXform = mat44f::identity();
    mat44f cameraProjMat = mat44f::identity();

    inData = block.inputValue ( attrInCameraInvMat, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);
    inData.asMatrix().get(m16f);
    cameraXform = mat44f::fromArray((float*)m16f);

    // create a frustum...

    float left = -1;
    float right = 1;
    float top = 1;
    float bottom = -1;
    float _near = -1;
    float _far = 1;

    float tx = -(right+left)/(right-left);
    float ty = -(top+bottom)/(top-bottom);
	float tz = -(_far+_near)/(_far-_near);

    cameraProjMat.m[0] = make_vec4f(2/(right-left),0,0,0);
    cameraProjMat.m[1] = make_vec4f(0,2/(top-bottom),0,0);
    cameraProjMat.m[2] = make_vec4f(0,0,-2/(_far-_near),0);
    cameraProjMat.m[3] = make_vec4f(tx,ty,tz,1);

    inData = block.inputValue ( attrGlTexDisplacementSize, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);
    int size = inData.asInt();

    float extents[3];
    extents[0] = std::max(64, size);
    extents[1] = std::max(64, size);
    extents[2] = 1;

    if (_pboId == 0 || extents[0] != _extents[0] || extents[1] != _extents[1] || extents[2] != _extents[2])
    {
        _extents[0] = extents[0];
        _extents[1] = extents[1];
        _extents[2] = extents[2];

        if (_pboId != 0)
        {
            glDeleteBuffers(1, &_pboId);
            _pboId = 0;
            glDeleteTextures(1, &_glTexId);
            _glTexId = 0;
            glDeleteTextures(1, &_glTexDepthId);
            _glTexDepthId = 0;
        }

        if (size > 0)
        {
            glGenBuffers( 1, &_pboId );
            glBindBuffer( GL_PIXEL_PACK_BUFFER, _pboId );
            glBufferData( GL_PIXEL_PACK_BUFFER, extents[0] * extents[1] * extents[2] * sizeof(float) * 4, NULL, GL_STREAM_READ );
            glBindBuffer( GL_PIXEL_PACK_BUFFER, 0 );

            _glTexId = Easy::gl::createTexture2D(GL_TEXTURE_2D, extents[0], extents[1], GL_RGBA32F, GL_RGBA);
            _glTexDepthId = Easy::gl::createTexture2D(GL_TEXTURE_2D, extents[0], extents[1], GL_DEPTH_COMPONENT24, GL_DEPTH_COMPONENT);

            glBindTexture(GL_TEXTURE_2D, 0);

            if (_fbo == 0)
                _fbo = new FrameBufferObject();

            _fbo->AttachTexture(GL_TEXTURE_2D, _glTexId, GL_COLOR_ATTACHMENT0_EXT);
            _fbo->AttachTexture(GL_TEXTURE_2D, _glTexDepthId, GL_DEPTH_ATTACHMENT_EXT);
            _fbo->IsValid();

            glCheckErrors();
        }
    }

    if (_pboId && _triMesh)
    {
        glPushAttrib(GL_VIEWPORT_BIT);
        _fbo->Bind();

        glClearColor(1.0, 0.0, 0.0, 0.0);
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        glViewport(0, 0, extents[0], extents[1]);

        glMatrixMode(GL_MODELVIEW);
        glPushMatrix();
        glLoadMatrixf((GLfloat*)&cameraXform);
        glMultMatrixf((GLfloat*)&geomXform);

        glMatrixMode(GL_PROJECTION);
        glPushMatrix();
        glLoadMatrixf((GLfloat*)&cameraProjMat);

        _program->enable();
        _program->setUniformMatrix4fv("modelToWorldMat", (GLfloat*)&geomXform, false);

        drawMesh(_triMesh);

        _program->disable();

        glMatrixMode(GL_PROJECTION);
        glPopMatrix();

        glMatrixMode(GL_MODELVIEW);
        glPopMatrix();

        glCheckErrors();

        // blit into pbo...
        //glReadBuffer(GL_FRONT);

        glBindBuffer(GL_PIXEL_PACK_BUFFER, _pboId);
        glReadPixels(0, 0, extents[0], extents[1], GL_RGBA, GL_FLOAT, 0);
        glBindBuffer(GL_PIXEL_PACK_BUFFER, 0);

        _fbo->Disable();

        glPopAttrib();

        glFlush();

        glCheckErrors();
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------------
MStatus NvParticlesParticleShape::_computeBounds(MDataBlock& block)
{
    MStatus status;

    MDataHandle hLow = block.outputValue( attrBoundsLow, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);
    MDataHandle hHigh = block.outputValue( attrBoundsHigh, &status );
    CHECK_MSTATUS_AND_RETURN_IT(status);

    float3 &low = hLow.asFloat3();
    float3 &high = hHigh.asFloat3();

    particleContainer->getBounds((float*)low, (float*)high);

    hLow.setClean();
    hHigh.setClean();

    return status;
}

//-----------------------------------------------------------------------------------
/// Ensure we are connected to the time attribute.
///
MStatus NvParticlesParticleShape::_ensureTimeConnection()
{
    MStatus status;
    MPlug timePlug( thisMObject(), attrCurrentTime );

    MSelectionList sList;
    MGlobal::getSelectionListByName("time1", sList);
    unsigned int nMatches = sList.length();
    if ( nMatches > 0 )
    {
        MObject timeDepObj;
        sList.getDependNode(0, timeDepObj);
        MFnDependencyNode timeDep( timeDepObj );
        MPlug outTimePlug = timeDep.findPlug("outTime",&status);
        CHECK_MSTATUS(status);
        if ( !timePlug.isConnected() )
        {
            Easy::NvParticles::printInfo("Connecting time attribute.");
            MDGModifier mod;
            mod.connect( outTimePlug, timePlug );
            mod.doIt();
            _timeIsConnected = true;
        }
        else
        {
            _timeIsConnected = true;
        }
    }

    return status;
}

//-----------------------------------------------------------------------------------
/// Ensure we are connected to something that updates the render.
///
MStatus NvParticlesParticleShape::_connectDummyToShear()
{
    MStatus status;

    MFnDagNode dagFn(thisMObject(), &status);
    CHECK_MSTATUS(status);
    MObject parent = dagFn.parent(0, &status);
    if(parent.isNull())
        return MS::kFailure;

    dagFn.setObject(parent);
    MPlug triggerPlug = dagFn.findPlug("shearXY",&status);
    //MPlug triggerPlug = dagFn.findPlug("selectHandleX",&status);

    CHECK_MSTATUS(status);
    MPlug dummyPlug = MPlug(thisMObject(), attrDummy);
    if(!dummyPlug.isConnected())
    {
        MDGModifier mod;
        mod.connect( dummyPlug, triggerPlug );
        mod.doIt();

        Easy::NvParticles::printInfo("Connecting output to shear attribute.");
        _dummyIsConnected = true;
    }
    return status;
}

//-----------------------------------------------------------------------------------
static MStatus getWorldMatrix( const MObject& thisNode, MMatrix &worldMatrix )
{
	MStatus status;

	MFnDependencyNode fnThisNode( thisNode );

	// get worldMatrix attribute.
	//
	MObject worldMatrixAttr = fnThisNode.attribute( "worldMatrix" );

	// build worldMatrix plug, and specify which element the plug refers to.
	// We use the first element(the first dagPath of this emitter).
	//
	MPlug matrixPlug( thisNode, worldMatrixAttr );
	matrixPlug = matrixPlug.elementByLogicalIndex( 0 );

	// Get the value of the 'worldMatrix' attribute
	//
	MObject matrixObject;
	status = matrixPlug.getValue( matrixObject );
	if( !status )
	{
		status.perror("getWorldPosition: get matrixObject");
		return( status );
	}

	MFnMatrixData worldMatrixData( matrixObject, &status );
	if( !status )
	{
		status.perror("NvParticlesParticleEmitter::getWorldPosition: get worldMatrixData");
		return( status );
	}

	worldMatrix = worldMatrixData.matrix( &status );
	if( !status )
	{
		status.perror("NvParticlesParticleEmitter::getWorldPosition: get worldMatrix");
		return( status );
	}

    return( status );
}

//-----------------------------------------------------------------------------------
/// Read any connected primitives and pass their information to the particleSolver.
/// n.b.
/// This is not as efficient as it could be as it is evaluated every update.
/// For example, we could run this method only if a primitive changes.
///
MStatus NvParticlesParticleShape::_readPrimitives()
{
    MStatus status = MS::kSuccess;

    // get the connected primitives...
    MObject thisObject(thisMObject());
    MPlug plugs(thisObject, attrPrimitives);
    unsigned int count = plugs.numElements();

    for (unsigned int i=0; i<count; ++i)
    {
        MPlugArray connections;
        if (!plugs[i].connectedTo(connections, true, false, &status))
            continue;
        CHECK_MSTATUS(status);
        if (connections.length() != 1)
            continue;

        MObject object = connections[0].node();

        MFnDagNode fnDagNode(object, &status);
        CHECK_MSTATUS(status);

        // ensure we are a valid node type.
        if(fnDagNode.typeId() != NvParticlesPrimitiveShape::id)
            continue;

        // get the matrix.
        MDagPath dagPath;
        CHECK_MSTATUS(fnDagNode.getPath(dagPath));
        MMatrix mxform = dagPath.inclusiveMatrix();
        float m16f[4][4];
        mxform.get(m16f);
        mat44f xform = mat44f::fromArray((float*)m16f);

        bool enabled = getBoolAttr(fnDagNode, "enable");

        std::string primitiveName(dagPath.partialPathName().asChar());

        if (enabled)
        {
            MFloatVector extent = getFloat3Attr(fnDagNode, "extents");
            bool interior = getBoolAttr(fnDagNode, "interior");
            int primitive = getShortAttr(fnDagNode, "primitive");
            int type = getShortAttr(fnDagNode, "type");

            int flags = 0;
            if(interior)
                flags |= 0x01;

            NvParticles::Primitive prim;
            prim.type = 0;
            prim.extents = vec3f::fromArray((float*)&extent);
            prim.xform = xform;
            prim.flags = flags;
            prim.prevXform = xform;

            if (type == NvParticlesPrimitiveShape_TYPE_COLLISION)
            {
				if (primitive == NvParticlesPrimitiveShape_PRIMITIVE_SPHERE)
				{
                    prim.type = Easy::NvParticles::Primitive::PRIMITIVE_SPHERE;
					particleSolver->addPrimitive(primitiveName, prim);
				}
				else if (primitive == NvParticlesPrimitiveShape_PRIMITIVE_BOX)
				{
                    prim.type = Easy::NvParticles::Primitive::PRIMITIVE_BOX;
					particleSolver->addPrimitive(primitiveName, prim);
				}
				else if (primitive == NvParticlesPrimitiveShape_PRIMITIVE_PLANE)
				{
                    prim.type = Easy::NvParticles::Primitive::PRIMITIVE_PLANE;
					particleSolver->addPrimitive(primitiveName, prim);
				}
				else if (primitive == NvParticlesPrimitiveShape_PRIMITIVE_CAPSULE)
				{
                    prim.type = Easy::NvParticles::Primitive::PRIMITIVE_CAPSULE;
					particleSolver->addPrimitive(primitiveName, prim);
				}
            }
            else
            {
                if(particleSolver->_primitives)
                    particleSolver->_primitives->remove(primitiveName);

                if(type == NvParticlesPrimitiveShape_TYPE_EMISSION)
                {
                    if (primitive == Easy::NvParticles::Primitive::PRIMITIVE_BOX)
                    {
                        particleContainer->emitBox(xform, particleSolver->getParticleSpacing(), 0, make_vec4f(1), particleSolver->_currentTime);
                    }
                }
            }
        }
        else
        {
            // if disabled, then remove...
            if(particleSolver->_primitives)
                particleSolver->_primitives->remove(primitiveName);
        }
    }

    return status;
}

//------------------------------------------------------------------------------------------
/// Read any connected fields and pass their information to the particles.
/// n.b.
/// This is not as efficient as it could be as it is evaluated every update.
/// For example, we could run this method only if a value in the Field changes.
///
MStatus NvParticlesParticleShape::_readFields()
{
    MStatus status = MS::kSuccess;

    // unlike primitives, we do not need any pre-existing data in the forces.
    // so we may as well clear them.
    particleSolver->_forces->clear();

    // get the connected fields...
    MObject thisObject(thisMObject());
    MPlug plugs(thisObject, attrFields);
    unsigned int count = plugs.numElements();

    for(unsigned int i=0; i<count; ++i)
    {
        unsigned int logicalIndex = plugs[i].logicalIndex();

        MPlugArray connections;
        if (!plugs[i].connectedTo(connections, true, false, &status))
            continue;
        CHECK_MSTATUS(status);
        if (connections.length() != 1)
            continue;

        MObject object = connections[0].node();

        MFnField fnField(object, &status);
        CHECK_MSTATUS(status);

        MDagPath dagPath;
        CHECK_MSTATUS(fnField.getPath(dagPath));

        MString type = fnField.typeName();
		MString name = dagPath.partialPathName();

        // collect base attributes...
        float magnitude = (float)fnField.magnitude();
        float attenuation = (float)fnField.attenuation();
        bool useMaxDistance = fnField.useMaxDistance();
        float maxDistance = (useMaxDistance)?(float)fnField.maxDistance():-1;

        /// TODO:
        /// should probably get ALL instances of this dagnode.
        MMatrix mat = dagPath.inclusiveMatrix();
        float m16[4][4];
        mat.get(m16);

        // get the origin from the matrix.
        MFloatVector origin(m16[3][0],m16[3][1],m16[3][2]);

        if(type == "gravityField")
        {
            MFnGravityField fnF(object);
            MFloatVector direction = fnF.direction();

            NvParticles::ForceData force;
            force.axis = vec3f::fromArray((float*)&direction[0]);
            force.origin = vec3f::fromArray((float*)&origin[0]);
            force.type = NvParticles::ForceData::FORCE_UNIFORM;
            force.magnitude = magnitude;
            force.maxDistance = maxDistance;
            force.attenuation = attenuation;
            particleSolver->addForce(name.asChar(), force);
        }
        else if(type == "uniformField")
        {
            MFnUniformField fnF(object);
            MFloatVector direction = fnF.direction();

            NvParticles::ForceData force;
            force.axis = vec3f::fromArray((float*)&direction[0]);
            force.origin = vec3f::fromArray((float*)&origin[0]);
            force.type = NvParticles::ForceData::FORCE_UNIFORM;
            force.magnitude = magnitude;
            force.maxDistance = maxDistance;
            force.attenuation = attenuation;
            particleSolver->addForce(name.asChar(), force);
        }
        else if(type == "radialField")
        {
            MFnRadialField fnF(object);
            double type = fnF.radialType();
            if(type != 1)
            {
                MessageOnce("nvParticles - \"radialType\" attribute must be 1: " + name + " = " + type);
                continue;
            }

            NvParticles::ForceData force;
            force.origin = vec3f::fromArray((float*)&origin[0]);
            force.type = NvParticles::ForceData::FORCE_RADIAL;
            force.magnitude = magnitude;
            force.maxDistance = maxDistance;
            force.attenuation = attenuation;
            particleSolver->addForce(name.asChar(), force);
        }
        else if(type == "vortexField")
        {
            MFnVortexField fnF(object);
            MFloatVector axis = fnF.axis();

            NvParticles::ForceData force;
            force.axis = vec3f::fromArray((float*)&axis[0]);
            force.origin = vec3f::fromArray((float*)&origin[0]);
            force.type = NvParticles::ForceData::FORCE_VORTEX;
            force.magnitude = magnitude;
            force.maxDistance = maxDistance;
            force.attenuation = attenuation;
            particleSolver->addForce(name.asChar(), force);
        }
        else if(type == "dragField")
        {
            MFnDragField fnF(object);
            MFloatVector axis(0,0,0);
			if(fnF.useDirection())
				axis = fnF.direction();

            NvParticles::ForceData force;
            force.axis = vec3f::fromArray((float*)&axis[0]);
            force.origin = vec3f::fromArray((float*)&origin[0]);
            force.type = NvParticles::ForceData::FORCE_DRAG;
            force.magnitude = magnitude;
            force.maxDistance = maxDistance;
            force.attenuation = attenuation;
            particleSolver->addForce(name.asChar(), force);
        }
        else if(type == "airField")
        {
            MFnAirField fnF(object);
			MFloatVector axis = fnF.direction() * magnitude;

            // use the magnitude to store the speed.
			magnitude = fnF.speed();

            double inheritVelocity = fnF.inheritVelocity();

			if (inheritVelocity > 0)
			{
				// get the velocity direction of the transform.
				/// CAVEAT:
				// This does not include angular velocity.
			}

			if (fnF.inheritRotation())
			{
				/// TODO:
				// Do we want to remove the scaling factor?

				// rotate axis by the rotation-matrix.
				MFloatMatrix m(m16);
				axis = axis * m;
			}

			if (fnF.enableSpread())
			{
	            MessageOnce("nvParticles - \"spread\" attribute is not supported from: " + name + " = " + type);
			}

            NvParticles::ForceData force;
            force.axis = vec3f::fromArray((float*)&axis[0]);
            force.origin = vec3f::fromArray((float*)&origin[0]);
            force.type = NvParticles::ForceData::FORCE_AIR;
            force.magnitude = magnitude;
            force.maxDistance = maxDistance;
            force.attenuation = attenuation;
            particleSolver->addForce(name.asChar(), force);
        }
        else
        {
			MessageOnce("nvParticles - Unsupported field type: " + name + " = " + type);
        }
    }

    return status;
}

//-----------------------------------------------------------------------------------
/// Get any new particles created by emitters and pass them to the particles.
///
MStatus NvParticlesParticleShape::_readEmitters( const MPlug& plug, MDataBlock& block )
{
    MStatus status = MS::kSuccess;

    MObject thisObject(thisMObject());
    MPlug plugs(thisObject, attrEmitters);
    unsigned int count = plugs.numElements();

    for (unsigned int i=0; i<count; ++i)
    {
        MPlugArray connections;
        if (!plugs[i].connectedTo(connections, true, false, &status))
            continue;

        CHECK_MSTATUS(status);
        if (connections.length() != 1)
            continue;

        MObject object = connections[0].asMObject();
        MFnArrayAttrsData fnEmitterData( object, &status );
        CHECK_MSTATUS_AND_RETURN_IT(status);

        //get the emitters arrays for debugging
        MStringArray emitterDataNames = fnEmitterData.list(&status);
        CHECK_MSTATUS_AND_RETURN_IT(status);

        // get the basics:
        MVectorArray emitterPositions = fnEmitterData.vectorArray( "position", &status );
        CHECK_MSTATUS_AND_RETURN_IT(status);
        MVectorArray emitterVelocities = fnEmitterData.vectorArray( "velocity", &status );
        CHECK_MSTATUS_AND_RETURN_IT(status);
        MDoubleArray emitterTimeInSteps = fnEmitterData.doubleArray( "timeInStep", &status );
        CHECK_MSTATUS_AND_RETURN_IT(status);

        // now get the optional arrays:
        MVectorArray emitterColors;
        MFnArrayAttrsData::Type arrayType;
        if(fnEmitterData.checkArrayExist( "rgbPP", arrayType))
        {
            assert(arrayType == MFnArrayAttrsData::kVectorArray);
            emitterColors = fnEmitterData.vectorArray( "rgbPP", &status );
            CHECK_MSTATUS_AND_RETURN_IT(status);
        }

        MDoubleArray emitterOpacity;
        if(fnEmitterData.checkArrayExist( "opacityPP", arrayType))
        {
            assert(arrayType == MFnArrayAttrsData::kDoubleArray);
            emitterOpacity = fnEmitterData.doubleArray( "opacityPP", &status );
            CHECK_MSTATUS_AND_RETURN_IT(status);
        }

        float currentTime = particleSolver->_currentTime;

        unsigned int n = emitterPositions.length();

        if (n > 0)
        {
            double* rgbPP = 0;
            double* opacityPP = 0;

            if(emitterColors.length() > 0)
                rgbPP = (double*)&emitterColors[0];

            if(emitterOpacity.length() == n)
                opacityPP = (double*)&emitterOpacity[0];

            particleContainer->emitBuffers(n, currentTime, (double*)&emitterPositions[0], (double*)&emitterVelocities[0], (double*)&emitterTimeInSteps[0], rgbPP, emitterColors.length(), opacityPP);
        }
    }
    return status;
}

//-----------------------------------------------------------------------------------
void* NvParticlesParticleShape::creator()
{
    return new NvParticlesParticleShape();
}

//-----------------------------------------------------------------------------------
void NvParticlesParticleShape::onNodeRemoved(MObject &obj, void* clientData)
{
    MFnDependencyNode fnNode(obj);
    NvParticlesParticleShape *pxNode = static_cast<NvParticlesParticleShape *>(fnNode.userNode());
    pxNode->_destroy();
}

//-----------------------------------------------------------------------------------
MStatus NvParticlesParticleShape::connectionBroken( const MPlug& plug, const MPlug& otherPlug, bool asSrc )
{
    MStatus status;

    const MPlug &root = plug.isChild() ? plug.parent() : plug;
    //MGlobal::displayInfo(MString("connectionBroken: dest=") + plug.name());

    if ( plug == attrEmitters )
    {
        MObject thisObj = thisMObject();
        MPlug plug( thisObj, attrEmitters );
    }
    else if ( plug == attrFields )
    {
        // this is unnecessary because we are clearing the forces every frame.
        // but let's keep it here for good measure.

        MObject srcNode = otherPlug.node(&status);
        CHECK_MSTATUS(status);

        MFnField fieldFn(srcNode, &status);
        CHECK_MSTATUS(status);

        if (status == MS::kSuccess)
        {
            MDagPath srcDagPath;
            CHECK_MSTATUS(fieldFn.getPath(srcDagPath));

            std::string name = srcDagPath.partialPathName().asChar();
            particleSolver->_forces->remove(name);
        }
    }
    else if ( plug == attrPrimitives )
    {
        MObject srcNode = otherPlug.node(&status);
        CHECK_MSTATUS(status);
        MFnDagNode srcNodeFn(srcNode, &status);

        if(srcNodeFn.typeId() == NvParticlesPrimitiveShape::id)
        {
            MDagPath srcDagPath;
            CHECK_MSTATUS(srcNodeFn.getPath(srcDagPath));

            /// BUG:
            /// If the primitive is renamed in Maya, then this will
            /// fail to remove it.

            std::string name = srcDagPath.partialPathName().asChar();
            particleSolver->_primitives->remove(name);
        }
    }

    return MPxNode::connectionBroken( plug, otherPlug, asSrc );
}

//-----------------------------------------------------------------------------------
void NvParticlesParticleShape::_destroy()
{
    delete particleSolver;
    particleSolver = 0;

    delete particleRenderer;
    particleRenderer = 0;

    delete particleContainer;
    particleContainer = 0;

    delete _fbo;
    _fbo = 0;
    delete _program;
    _program = 0;

    if (_pboId != 0)
    {
        glDeleteBuffers(1, &_pboId);
        _pboId = 0;
        glDeleteTextures(1, (GLuint*)&_glTexId);
        _glTexId = 0;
        glDeleteTextures(1, (GLuint*)&_glTexDepthId);
        _glTexDepthId = 0;
    }

    delete _triMesh;
    _triMesh = 0;
}

//-----------------------------------------------------------------------------------
MStatus NvParticlesParticleShape::_computeOutputArrayAttrs( const MPlug& plug, MDataBlock& block )
{
    MStatus status = MS::kSuccess;

    MDataHandle hdl = block.outputValue(plug.attribute(), &status);
    CHECK_MSTATUS_AND_RETURN_IT(status);

	MFnArrayAttrsData fn;
    //MObject outputObject = fn.create(&status);
	MObject obj = hdl.data();

	CHECK_MSTATUS_AND_RETURN_IT(status);
	status = fn.setObject(obj);
	CHECK_MSTATUS_AND_RETURN_IT(status);

    /// CAVEAT:
	// nParticles's cache-array requires "id", "position", and "count" arrays.

    /// TODO:
    // I think we want to merge all containers into one big maya array.

    NvParticles::ParticleContainer* cit = particleContainer;

    if (1)
	{
        NvParticles::ParticleBuffer* positionBuffer = cit->getBuffer("position");
        if (positionBuffer && positionBuffer->getUpdateId() != cit->_currentUpdateState)
        {
            Easy::NvParticles::printError(positionBuffer->spec.name + " buffer is not current.");
            positionBuffer = 0;
        }

        NvParticles::ParticleBuffer* birthTimeBuffer = cit->getBuffer("birthTime");
        if (birthTimeBuffer && birthTimeBuffer->getUpdateId() != cit->_currentUpdateState)
        {
            Easy::NvParticles::printError(birthTimeBuffer->spec.name + " buffer is not current.");
            birthTimeBuffer = 0;
        }

        NvParticles::ParticleBuffer* velocityBuffer = cit->getBuffer("velocity");
        if (velocityBuffer && velocityBuffer->getUpdateId() != cit->_currentUpdateState)
        {
            Easy::NvParticles::printError(velocityBuffer->spec.name + " buffer is not current.");
            velocityBuffer = 0;
        }

        NvParticles::ParticleBuffer* idBuffer = cit->getBuffer("id");
        if (idBuffer && idBuffer->getUpdateId() != cit->_currentUpdateState)
        {
            Easy::NvParticles::printError(idBuffer->spec.name + " buffer is not current.");
            idBuffer = 0;
        }

        NvParticles::ParticleBuffer* colorBuffer = cit->getBuffer("color");
        if (colorBuffer && colorBuffer->getUpdateId() != cit->_currentUpdateState)
        {
            Easy::NvParticles::printError(colorBuffer->spec.name + " buffer is not current.");
            colorBuffer = 0;
        }

	    MDoubleArray countArray = fn.doubleArray("count", &status);
	    CHECK_MSTATUS_AND_RETURN_IT(status);
	    countArray.setLength(1);
	    countArray[0] = cit->numParticles;

	    if (positionBuffer)
	    {
		    MVectorArray outArray = fn.vectorArray("position", &status);
		    CHECK_MSTATUS_AND_RETURN_IT(status);
		    outArray.setLength(cit->numParticles);

            float* ptr = (float*)positionBuffer->hostPointer();

		    for (int i=0; i<cit->numParticles; ++i)
		    {
			    MFloatVector fv(&ptr[i*4]);
			    MVector vec(fv);
			    outArray[i] = vec;
		    }
	    }

	    if (birthTimeBuffer)
	    {
		    MDoubleArray outArray = fn.doubleArray("birthTime", &status);
		    CHECK_MSTATUS_AND_RETURN_IT(status);
		    outArray.setLength(cit->numParticles);

            float* ptr = (float*)birthTimeBuffer->hostPointer();

		    for (int i=0; i<cit->numParticles; ++i)
		    {
			    outArray[i] = ptr[i];
		    }
	    }

	    if (colorBuffer)
	    {
		    MVectorArray outArray = fn.vectorArray("rgbPP", &status);
		    CHECK_MSTATUS_AND_RETURN_IT(status);
		    outArray.setLength(cit->numParticles);

            float* ptr = (float*)colorBuffer->hostPointer();

		    for (int i=0; i<cit->numParticles; ++i)
		    {
			    MFloatVector fv(&ptr[i*4]);
			    MVector vec(fv);
			    outArray[i] = vec;
		    }
	    }

	    if (colorBuffer)
	    {
		    MDoubleArray outArray = fn.doubleArray("opacityPP", &status);
		    CHECK_MSTATUS_AND_RETURN_IT(status);
		    outArray.setLength(cit->numParticles);

            float* ptr = (float*)colorBuffer->hostPointer();

		    for (int i=0; i<cit->numParticles; ++i)
		    {
			    outArray[i] = ptr[i*4+3];
		    }
	    }

	    if (idBuffer)
	    {
		    MIntArray outArray = fn.intArray("id", &status);
		    CHECK_MSTATUS_AND_RETURN_IT(status);
		    outArray.setLength(cit->numParticles);

            int* ptr = (int*)idBuffer->hostPointer();

		    for (int i=0; i<cit->numParticles; ++i)
		    {
			    outArray[i] = ptr[i];
		    }
	    }

	    if (velocityBuffer)
	    {
		    MVectorArray outArray = fn.vectorArray("velocity", &status);
		    CHECK_MSTATUS_AND_RETURN_IT(status);
		    outArray.setLength(cit->numParticles);

            float* ptr = (float*)velocityBuffer->hostPointer();

		    for (int i=0; i<cit->numParticles; ++i)
		    {
			    MFloatVector fv(&ptr[i*4]);
			    MVector vec(fv);
			    outArray[i] = vec;
		    }
	    }
    }

	// do we need to do this?
	//fn.set(outArray);

	hdl.set(obj);

    return status;
}

//-----------------------------------------------------------------------------------
void NvParticlesParticleShape::_updateMeshData(const MObject& meshObj)
{
    MStatus status;

    if (_triMesh == 0)
    {
        _triMesh = _buildMeshData(meshObj);
        return;
    }

    MFnMesh meshFn(meshObj);

	int nv = meshFn.numVertices();
    bool rebuild = false;

    if (nv != _triMesh->nVertices)
    {
        delete _triMesh;
        _triMesh = _buildMeshData(meshObj);
        return;
    }

    const float* rawPoints = meshFn.getRawPoints(&status);
	CHECK_MSTATUS( status );

	memcpy(&_triMesh->vertices[0], rawPoints, nv*3*sizeof(float));
}

//-----------------------------------------------------------------------------------
TriangleMesh* NvParticlesParticleShape::_buildMeshData(const MObject& meshObj)
{
	MStatus status;
	MFnMesh meshFn(meshObj);

	int nv = meshFn.numVertices();
	if(nv == 0)
		return NULL;

    ScopedTimer _st("_buildMeshData");

	TriangleMesh* triMesh = new TriangleMesh;

    triMesh->nVertices = nv;
    triMesh->vertices.resize(nv*3);
    triMesh->normals.resize(nv*3, 0);

	MItMeshPolygon polyIt(meshObj, &status);
	CHECK_MSTATUS( status );

	MPointArray mpoints;
	MIntArray mindices;

    bool isLocked = true;
    MMutexLock fCriticalSection;
	fCriticalSection.lock();

	triMesh->nTriangles = 0;

	for (; !polyIt.isDone(); polyIt.next())
	{
		if (polyIt.hasValidTriangulation())
		{
            //  Get the number of triangles in the current polygon.
            int numPolyTriangles = 0;
            polyIt.numTriangles(numPolyTriangles);

			for(int ti=0; ti<numPolyTriangles; ++ti)
			{
				status = polyIt.getTriangle(ti, mpoints, mindices, MSpace::kObject);

				for(int tvi=0; tvi<mindices.length(); ++tvi)
				{
                    MVector mnormal;
                    status = polyIt.getNormal(tvi, mnormal, MSpace::kObject);

                    int vindex = mindices[tvi];
					triMesh->indices.push_back(vindex);

                    // modify the vertex to be in world-space!
                    triMesh->vertices[vindex*3+0] = mpoints[tvi].x;
                    triMesh->vertices[vindex*3+1] = mpoints[tvi].y;
                    triMesh->vertices[vindex*3+2] = mpoints[tvi].z;

                    triMesh->normals[vindex*3+0] += mnormal.x;
                    triMesh->normals[vindex*3+1] += mnormal.y;
                    triMesh->normals[vindex*3+2] += mnormal.z;
				}
            }

			triMesh->nTriangles += numPolyTriangles;

            //  If hasValidTriangulation() was going to triangulate the
            //  mesh it will have done so by now. Subsequent calls will
            //  use the existing triangulation so it's safe to remove the lock now.
            if (isLocked)
			{
                fCriticalSection.unlock();
                isLocked = false;
            }
        }
	}

    for (int vindex=0; vindex<nv; ++vindex)
    {
        float nx = triMesh->normals[vindex*3+0];
        float ny = triMesh->normals[vindex*3+1];
        float nz = triMesh->normals[vindex*3+2];
        float nlen = sqrtf(nx*nx+ny*ny+nz*nz);
        triMesh->normals[vindex*3+0] = nx / nlen;
        triMesh->normals[vindex*3+1] = ny / nlen;
        triMesh->normals[vindex*3+2] = nz / nlen;
    }

    return triMesh;
}


//-----------------------------------------------------------------------------------
static void drawMesh(TriangleMesh* triMesh)
{
	MStatus status;

    int nTriangles = 0;
    int nVertices = 0;

    if (triMesh == 0)
        return;

    nVertices = triMesh->nVertices;
    nTriangles = triMesh->nTriangles;
    float* posPtr = &triMesh->vertices[0];
    float* normalPtr = &triMesh->normals[0];
    int* idx = &triMesh->indices[0];

    glMatrixMode(GL_MODELVIEW);

	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, 0, posPtr);

	glEnableClientState(GL_NORMAL_ARRAY);
	glNormalPointer(GL_FLOAT, 0, normalPtr);

/*
    MDagPathArray dagPaths;
    meshFn.getAllPaths(dagPaths);

    for (unsigned int j = 0; j < dagPaths.length(); j++ )
    {
        const MDagPath& dagPath = dagPaths[j];

        MMatrix mxform = dagPath.inclusiveMatrix();
        float m16f[4][4];
        mxform.get(m16f);
        mat44f xform = mat44f::fromArray((float*)m16f);

        glPushMatrix();
        glMultMatrixf((GLfloat*)&xform);
*/

	    if(idx)
	    {
		    glDrawElements(GL_TRIANGLES, nTriangles*3, GL_UNSIGNED_INT, idx);
	    }
	    else
	    {
		    glDrawArrays(GL_POINTS, 0, nVertices);
	    }
/*
        glPopMatrix();
    }
*/

	glDisableClientState(GL_VERTEX_ARRAY);
    glDisableClientState(GL_NORMAL_ARRAY);
}

//-----------------------------------------------------------------------------------
