/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#ifndef NVPARTICLESPARTICLESHAPE_H_INCLUDED
#define NVPARTICLESPARTICLESHAPE_H_INCLUDED

#include "gl_utils.h"
#include "math_utils.h"

#include <maya/MStatus.h>
#include <maya/MObject.h>
#include <maya/MPxLocatorNode.h>
#include <maya/MPxSurfaceShape.h>
#include <maya/MDGMessage.h>
#include <maya/MStringArray.h>

#include "NvParticlesTypes.h"

#include <vector>

namespace Easy
{
namespace NvParticles
{
class Manager;
class ParticleContainer;
class ParticleRenderer;
class ParticleSolver;
}
}

//-----------------------------------------------------------------------------------
class MPlug;
class MDataBlock;

//-----------------------------------------------------------------------------------
class TriangleMesh
{
public:
	Easy::vec3f minCorner;
	Easy::vec3f maxCorner;
	int nVertices;
	int nTriangles;
	std::vector<float> vertices;
    std::vector<float> normals;
	std::vector<int> indices;

	TriangleMesh()
	{
		nVertices = 0;
		nTriangles = 0;
	}
};

//-----------------------------------------------------------------------------------
class NvParticlesParticleShape : public MPxSurfaceShape
{
    friend class NvParticlesParticleShapeUI;

public:
	static void* creator();
	static MStatus initialize();

    NvParticlesParticleShape();
	virtual ~NvParticlesParticleShape();

    virtual MStatus connectionBroken( const MPlug& plug, const MPlug& otherPlug, bool asSrc );

    virtual void postConstructor();
    virtual MStatus compute( const MPlug& plug, MDataBlock& block );

    virtual bool isBounded() const;
    virtual MBoundingBox boundingBox() const;

	virtual bool isTransparent() const;
	virtual bool drawLast() const;

    virtual MStatus setDependentsDirty(MPlug const & inPlug, MPlugArray  & affectedPlugs);

    void glDraw(const MDagPath& path, int width, int height, float fov, const MMatrix& modelViewMat, bool drawDiagnostics);

    void drawView( M3dView & view, const MDagPath & path, M3dView::DisplayStyle style, M3dView::DisplayStatus displaystatus );

protected:

    MStatus _updateOcean(MDataBlock& block);

    MStatus _updateParticlesFromAttributes(MDataBlock& block);

	MStatus _readEmitters(const MPlug& plug, MDataBlock& block);

	MStatus _readFields();

    MStatus _readPrimitives();

    MStatus _ensureTimeConnection();
    bool _timeIsConnected;

    MStatus _connectDummyToShear();
    bool _dummyIsConnected;

    void _destroy();

    MStatus _computeOutputArrayAttrs( const MPlug& plug, MDataBlock& block );

    MStatus _computeBounds(MDataBlock& block);

    void _updateRenderer();

public:
	static MTypeId id;
    static MCallbackId nodeRemovedCallback;
    static MCallbackId nodePreConnectionCallback;

    static void onNodeRemoved(MObject &obj, void* clientData);
    static void onNodePreConnection(MPlug &srcPlug, MPlug &destPlug, bool made, void *clientData);

    // attributes:

    static std::vector< std::pair<MObject, Easy::NvParticles::ParameterSpecPtr> > _solverAttributes;

	static MObject attrDebugLevel;
    static MObject attrEnable;
    static MObject attrMaxParticles;

    static MObject attrCurrentTime;
    static MObject attrTimeStep;
    static MObject attrStartTime;

    // emitter, field, and primitive inputs:
	static MObject attrEmitters;
    static MObject attrFields;
	static MObject attrPrimitives;

    // outputs (for emitters)
    static MObject attrIsFull;
    static MObject attrDeltaTime;
    static MObject attrSeed;
    static MObject attrInheritFactor;

    // rendering:
    static MObject attrRendererType;
    static MObject attrRendererRadius;
    static MObject attrRendererRadiusFactor;
    static MObject attrRendererColor;
    static MObject attrRendererPointSize;
    static MObject attrRendererDepthSort;
    static MObject attrRendererUseLighting;
	static MObject attrRendererStreakWidth;
	static MObject attrRendererStreakLength;
	static MObject attrRendererUseColor;

    // diagnostics:
    static MObject attrDrawPrimitives;
    static MObject attrDrawGrid;
    static MObject attrDrawBounds;
    static MObject attrDrawDebugTextures;

	// outputs:
	static MObject attrOutArrayAttrs;

    // internal:
    static MObject attrBoundsLow;
    static MObject attrBoundsHigh;
	static MObject attrDummy;
    static MObject attrLastTime;
    static MObject attrNumParticles;

    static MObject attrSolverEnable;
   
    static MObject attrGlTexDisplacementSize;
	static MObject attrInGeometry;
    static MObject attrInGeometryMat;
    static MObject attrInCameraInvMat;

    static MObject cudaDeviceAttr;

    Easy::NvParticles::ParticleContainer* particleContainer;
    Easy::NvParticles::ParticleSolver* particleSolver;
    Easy::NvParticles::ParticleRenderer* particleRenderer;

    // Viewport 2.0 Draw Registration...
	static MString drawDbClassification;
	static MString drawRegistrantId;

private:

	// This is set to true after the postConstructor.
	bool _isConstructed;
	Easy::NvParticles::Manager* _nvParticles;

    // for building the ocean geometry:

    TriangleMesh* _buildMeshData(const MObject& meshObj);
    void _updateMeshData(const MObject& meshObj);

    GLSLProgram* _program;
    FrameBufferObject* _fbo;
    GLuint _glTexDepthId;
    GLuint _glTexId;
    GLuint _pboId;
    float _extents[3];
    TriangleMesh* _triMesh;
};
//-----------------------------------------------------------------------------------

#endif // NVPARTICLESPARTICLESHAPE_H_INCLUDED
