/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#include "NvParticlesParticleShapeUI.h"
#include "NvParticlesParticleShape.h"
#include <maya/MMaterial.h>
#include <maya/MDagPath.h>
#include <maya/MSelectionMask.h>
#include <maya/MSelectionList.h>


#include <maya/MTime.h>
#include <maya/MMatrix.h>
#include <maya/MFnCamera.h>

#include "NvParticlesProfiler.h"
#include "maya_utils.h"
#include "NvParticlesParticleRenderer.h"
#include "NvParticlesParticleModifier.h"

using namespace Easy;
using std::cerr;
using std::cout;

//-----------------------------------------------------------------------------------
#define LEAD_COLOR				18	// green
#define ACTIVE_COLOR			15	// white
#define ACTIVE_AFFECTED_COLOR	8	// purple
#define DORMANT_COLOR			4	// blue
#define HILITE_COLOR			17	// pale blue

//-----------------------------------------------------------------------------------
NvParticlesParticleShapeUI::NvParticlesParticleShapeUI() {}
NvParticlesParticleShapeUI::~NvParticlesParticleShapeUI() {}

//-----------------------------------------------------------------------------------
void* NvParticlesParticleShapeUI::creator()
{
	return new NvParticlesParticleShapeUI();
}

//-----------------------------------------------------------------------------------
/* override */
void NvParticlesParticleShapeUI::getDrawRequests( const MDrawInfo & info,
							 bool /*objectAndActiveOnly*/,
							 MDrawRequestQueue & queue )
{
	// are we displaying particles?
	if ( ! info.objectDisplayStatus( M3dView::kDisplayDynamics ) )
		return;

	MDrawRequest request = info.getPrototype( *this );
	NvParticlesParticleShape* shapeNode = (NvParticlesParticleShape*)surfaceShape();

    /// er.... should we pass something more lightweight than the shape ptr? (but why?)
	MDrawData data;
	getDrawData( NULL, data );

    // build the request
	request.setDrawData( data );
	request.setDrawLast(shapeNode->drawLast());
	request.setIsTransparent(shapeNode->isTransparent());

/*
	switch ( info.displayStyle() )
	{
		case M3dView::kWireFrame :
		case M3dView::kGouraudShaded :
		case M3dView::kFlatShaded :
			//request.setToken( kDrawFlatShaded );
 			getDrawRequestsShaded( request, info, queue, data );
			queue.add( request );
			break;
		default:
			break;
	}
*/
	switch ( info.displayStyle() )
	{
		case M3dView::kWireFrame :
			getDrawRequestsWireframe( request, info );
			queue.add( request );
			break;

		case M3dView::kGouraudShaded :
			request.setToken( kDrawSmoothShaded );
			getDrawRequestsShaded( request, info, queue, data );
			queue.add( request );
			break;

		case M3dView::kFlatShaded :
			request.setToken( kDrawFlatShaded );
 			getDrawRequestsShaded( request, info, queue, data );
			queue.add( request );
			break;

		default:
			break;
	}
}

//-----------------------------------------------------------------------------------
void NvParticlesParticleShapeUI::draw( const MDrawRequest & request, M3dView & view ) const
{
	MDrawData data = request.drawData();

    /// am I allowed to get the shape here (or must I use the drawdata)?
	//quadricGeom * geom = (quadricGeom*)data.geometry();
    NvParticlesParticleShape* shapeNode = (NvParticlesParticleShape*)surfaceShape();

	int token = request.token();
	bool drawTexture = false;

	view.beginGL();

	if ( (token == kDrawSmoothShaded) || (token == kDrawFlatShaded) )
	{
#if		defined(SGI) || defined(MESA)
		glEnable( GL_POLYGON_OFFSET_EXT );
#else
		glEnable( GL_POLYGON_OFFSET_FILL );
#endif

		MMaterial material = request.material();
		material.setMaterial( request.multiPath(), request.isTransparent() );

		drawTexture = material.materialIsTextured();
		if ( drawTexture )
		{
		    glEnable(GL_TEXTURE_2D);
			material.applyTexture( view, data );
		}

		shapeNode->drawView(view, request.multiPath(), request.displayStyle(), request.displayStatus());
	}
    else
    {
        shapeNode->drawView(view, request.multiPath(), request.displayStyle(), request.displayStatus());
    }

	view.endGL();
}

//-----------------------------------------------------------------------------------
/* override */
bool NvParticlesParticleShapeUI::select( MSelectInfo &selectInfo,
							 MSelectionList &selectionList,
							 MPointArray &worldSpaceSelectPts ) const
{
    /// BUG:
    /// We must ignore selection inside viewport, because it stalls Maya.
    /// (Presumably because of bbox; maybe the shearing)
    return false;

	MSelectionMask priorityMask( MSelectionMask::kSelectObjectsMask );
	MSelectionList item;
	item.add( selectInfo.selectPath() );
	MPoint xformedPt;
	selectInfo.addSelection( item, xformedPt, selectionList, worldSpaceSelectPts, priorityMask, false );
	return true;
}

//-----------------------------------------------------------------------------------
void NvParticlesParticleShapeUI::getDrawRequestsWireframe( MDrawRequest& request, const MDrawInfo& info )
{
	request.setToken( kDrawWireframe );

	M3dView::DisplayStatus displayStatus = info.displayStatus();
	M3dView::ColorTable activeColorTable = M3dView::kActiveColors;
	M3dView::ColorTable dormantColorTable = M3dView::kDormantColors;
	switch ( displayStatus )
	{
		case M3dView::kLead :
			request.setColor( LEAD_COLOR, activeColorTable );
			break;
		case M3dView::kActive :
			request.setColor( ACTIVE_COLOR, activeColorTable );
			break;
		case M3dView::kActiveAffected :
			request.setColor( ACTIVE_AFFECTED_COLOR, activeColorTable );
			break;
		case M3dView::kDormant :
			request.setColor( DORMANT_COLOR, dormantColorTable );
			break;
		case M3dView::kHilite :
			request.setColor( HILITE_COLOR, activeColorTable );
			break;
		default:
			break;
	}
}

//-----------------------------------------------------------------------------------
void NvParticlesParticleShapeUI::getDrawRequestsShaded( MDrawRequest& request,
											const MDrawInfo& info,
											MDrawRequestQueue& queue,
											MDrawData& data )
{
	// Need to get the material info
	//
	MDagPath path = info.multiPath();
	M3dView view = info.view();
	MMaterial material = MPxSurfaceShapeUI::material( path );
	M3dView::DisplayStatus displayStatus = info.displayStatus();

	// Evaluate the material and if necessary, the texture.
	//
	if ( ! material.evaluateMaterial( view, path ) )
	{
		cerr << "Couldnt evaluate\n";
	}

	bool drawTexture = true;
	if ( drawTexture && material.materialIsTextured() )
	{
		material.evaluateTexture( data );
	}

	request.setMaterial( material );

	bool materialTransparent = false;
	material.getHasTransparency( materialTransparent );
	if ( materialTransparent ) {
		request.setIsTransparent( true );
	}
/*
	// create a draw request for wireframe on shaded if
	// necessary.
	//
	if ( (displayStatus == M3dView::kActive) ||
		 (displayStatus == M3dView::kLead) ||
		 (displayStatus == M3dView::kHilite) )
	{
		MDrawRequest wireRequest = info.getPrototype( *this );
		wireRequest.setDrawData( data );
		getDrawRequestsWireframe( wireRequest, info );
		wireRequest.setToken( kDrawWireframeOnShaded );
		wireRequest.setDisplayStyle( M3dView::kWireFrame );
		queue.add( wireRequest );
	}*/
}


//-----------------------------------------------------------------------------------
//! This function takes in a screen x and y value and converts it to values
//! on the near and far clipping planes.
//! (Taken from Sony Imageworks's: spReticle)
static MPoint getPoint(float x, float y, M3dView & view, const MMatrix& worldInvMat)
{
    MPoint p, ncp, fcp;
    view.viewToWorld(int(x), int(y), ncp, fcp);
    MVector v = fcp - ncp;
    v.normalize();
    p = (ncp + v) * worldInvMat;
    return p;
}

//-----------------------------------------------------------------------------------
static void drawText(MString text, double tx, double ty,
                             MColor textColor, M3dView::TextPosition textAlign,
                             M3dView & view, const MMatrix& worldInvMat)
{
    // Turn off z-depth test
    glDisable( GL_DEPTH_TEST );
    glDepthMask( GL_FALSE );

    MPoint textPos = getPoint((float)tx, (float)ty, view, worldInvMat);

    //STDERR3(tx, ty, textPos);

    glColor4f( textColor.r, textColor.g, textColor.b, textColor.a );
    view.drawText(text, textPos, textAlign);

    // Turn on z-depth test
    glDepthMask( GL_TRUE );
    glEnable( GL_DEPTH_TEST );
}

//-----------------------------------------------------------------------------------
static void recurse_profile(M3dView & view, const MMatrix& worldInvMat, Easy::Profiler::ProfileIterator *iter,int global_xoffset, int &yoffset,int indent,float scale,int numFrames)
{
    int xoffset;
    for (;iter->IsDone() == false;iter->Next())
    {
        if (iter->Calls() == 0)
            continue;

        float thisscale = 1.f;
        if (iter->ParentElapsedTime() > 0.f)
            thisscale = iter->ElapsedTime()/iter->ParentElapsedTime();

        // percentage shows total time (over all functions!)
        int percentage = (int)(100.f*thisscale*scale);

        float count = numFrames;
        FORCE_MAX(count,1);

        float averageTimePerCall = float(iter->ElapsedTime()*1000.f)/iter->Calls();
        float timePerFrame = float(iter->ElapsedTime()*1000.f)/count;
        int callsPerFrame = int(iter->Calls()/count);


        const int NAME_XOFFSET=0+global_xoffset;
        const int GRAPH_XOFFSET=256+global_xoffset;
        const int INDENT_PIXELS=20;
        const int GRAPH_SIZE=30;
        const int YSPACING=16;


        MColor col(1*thisscale,1,1,1);

        xoffset = NAME_XOFFSET+indent*INDENT_PIXELS;
        drawText(iter->Name(), xoffset, yoffset, col, M3dView::kLeft, view, worldInvMat);

        xoffset = GRAPH_XOFFSET;
        //Easy::Gl::DrawSolidRectangle(xoffset,yoffset, xoffset+GRAPH_SIZE*(thisscale*scale),yoffset+10);

        xoffset = GRAPH_XOFFSET+GRAPH_SIZE+INDENT_PIXELS;
        std::string t = Easy::Stringf("calls=%05d time=%05fms avg/frame=%05fms -- %03d%%", callsPerFrame, timePerFrame, averageTimePerCall, percentage);
        drawText(t.c_str(), xoffset, yoffset, col, M3dView::kLeft, view, worldInvMat);

        yoffset -= YSPACING;

        Easy::Profiler::ProfileIterator iter2(iter->ThisNode());
        recurse_profile(view, worldInvMat, &iter2, global_xoffset,yoffset,indent+1,thisscale*scale,numFrames);
    }
}

//-----------------------------------------------------------------------------------
static void drawProfile(M3dView& view, const MMatrix& worldInvMat)
{
    Easy::Profiler::ProfileIterator *iter = NvParticles::Profiler::getSingleton().CreateIterator();
	int numFrames = NvParticles::Profiler::getSingleton().ElapsedFrames();

    if (!iter->IsDone())
    {
        int startx = 32;
        int starty = 48;
        int y = view.portHeight()-starty;
        recurse_profile(view, worldInvMat, iter,startx,y,0,1,numFrames);
    }

    delete iter;
}

//-----------------------------------------------------------------------------------
void NvParticlesParticleShape::glDraw( const MDagPath& path, int width, int height, float fov, const MMatrix& modelViewMat, bool drawDiagnostics)
{
    gl::initialize();

    MStatus status;

    // this ensures we are connected to time.
    if ( !_timeIsConnected )
        _ensureTimeConnection();

    MObject thisNode = thisMObject();
    MFnDependencyNode nodeFn(thisNode);

    MPlug plug(thisNode, attrDummy);
    plug.asFloat();

    plug.setAttribute(attrCurrentTime);
    float frame = plug.asMTime().as( MTime::uiUnit() );

    plug.setAttribute(attrEnable);
    bool enabled = plug.asBool();
    if (!enabled)
        return;

    // pass these to the particles object for its diagnostic render...

    plug.setAttribute(attrDrawBounds);
    bool drawBounds = plug.asBool();

    plug.setAttribute(attrDrawPrimitives);
    bool drawPrimitives = plug.asBool();

    plug.setAttribute(attrDrawGrid);
    bool drawGrid = plug.asBool();

    plug.setAttribute(attrRendererType);
    MString rendererType = plug.asString();

    glCheckErrors();

    // get rid of the particles transformation; they should always be in world-space...
    MMatrix modelMatInv = path.inclusiveMatrixInverse();
    MMatrix viewMat = modelMatInv * modelViewMat;
    float viewMat16[4][4];
    viewMat.get(viewMat16);

    glPushAttrib( GL_ALL_ATTRIB_BITS);
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadMatrixf((float*)viewMat16);

    // ensure we have a particleRenderer object...
    if (!particleRenderer)
    {
        particleRenderer = new NvParticles::ParticleRenderer();
    }

    // draw particles
    if (rendererType != "" && particleRenderer->setType(rendererType.asChar()))
    {
        // setup the renderer parameters...

        NvParticles::Parameters parameters;

        parameters["environment"] = 0;

        if (rendererType == "points")
        {
            plug.setAttribute(attrRendererPointSize);
            parameters["renderer_pointSize"].setInt(plug.asInt());

            plug.setAttribute(attrRendererStreakLength);
            parameters["renderer_streakLength"].setFloat(plug.asFloat());

			plug.setAttribute(attrRendererStreakWidth);
            parameters["renderer_streakWidth"].setFloat(plug.asFloat());

			plug.setAttribute(attrRendererUseColor);
            parameters["renderer_useColor"].setBool(plug.asBool());
		}
        else if (rendererType == "spheres")
        {
            plug.setAttribute(attrRendererRadius);
            parameters["renderer_absoluteRadius"].setFloat(plug.asFloat());

            plug.setAttribute(attrRendererRadiusFactor);
            parameters["renderer_radiusFactor"].setFloat(plug.asFloat());

            plug.setAttribute(attrRendererUseLighting);
            parameters["renderer_useLighting"].setBool(plug.asBool());
        }

        // it's not the most efficient to call this every frame,
        // but there is logic in here to avoid unnecessary reallocation.
        particleRenderer->resize(width, height, fov);

        particleRenderer->readColorTexture();
        particleRenderer->readDepthTexture();

        vec4f renderColor = parameters.asVector4("renderer_color", make_vec4f(0.0f, 0.8f, 1.0f,1));

        glColor4f(renderColor.x, renderColor.y, renderColor.z, 1.0);

		mat44f mat;
		glGetFloatv(GL_MODELVIEW_MATRIX, (GLfloat*)&mat);
		parameters["modelViewMatrix"] = mat;
		glGetFloatv(GL_PROJECTION_MATRIX, (GLfloat*)&mat);
		parameters["projectionMatrix"] = mat;

        particleRenderer->updateParameters(parameters);

        particleSolver->render(particleRenderer, drawBounds, drawGrid, drawPrimitives);

        glColor4f(1,1,1,1);
    }
    else
    {
        MString msg;
        msg.format("Unknown particle-renderer: ^1s", rendererType);
        MessageOnce(msg);
    }

    if(1) // overlay
    {
        plug.setAttribute(attrDrawDebugTextures);
        float debugTextureOpacity = plug.asFloat();
        if (debugTextureOpacity > 0)
        {
            glColor4f(1.f, 1.f, 1.f, debugTextureOpacity);

            // screen projection:
            glMatrixMode(GL_PROJECTION);
            glPushMatrix();
            glLoadIdentity();
            int w = std::max(width,64);
            int h = std::max(height,64);
            glOrtho(0, w, 0, h, -1, 1);

            glMatrixMode(GL_MODELVIEW);
            glPushMatrix();
            glLoadIdentity();

            if (particleRenderer)
                particleRenderer->renderDebugTextures(0, 0, 0.10f);

            glActiveTexture(GL_TEXTURE0);
            glMatrixMode(GL_TEXTURE);
            glPushMatrix();
            glLoadIdentity();
            glEnable(GL_TEXTURE_2D);
            glDisable(GL_BLEND);
            glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
            glBindTexture(GL_TEXTURE_2D, _glTexId);

            glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
            float sx = _extents[0], sy = _extents[1];
            float scale = 0.5;

            glBegin(GL_QUADS);
            glTexCoord2f(0,0);
            glVertex3f(width-sx*scale, 0,0);
            glTexCoord2f(1,0);
            glVertex3f(width, 0,0);
            glTexCoord2f(1,1);
            glVertex3f(width, sy*scale,0);
            glTexCoord2f(0,1);
            glVertex3f(width-sx*scale, sy*scale,0);
            glEnd();

            glBindTexture(GL_TEXTURE_2D, 0);
            glDisable(GL_TEXTURE_2D);
            glMatrixMode(GL_TEXTURE);
            glPopMatrix();
            glGetError();

            glMatrixMode(GL_PROJECTION);
            glPopMatrix();

            glMatrixMode(GL_MODELVIEW);
            glPopMatrix();

            glCheckErrors();
        }
    }

    ///glClientActiveTexture(GL_TEXTURE0);

    glPopMatrix(); // inverse worldXform
    glPopAttrib();
    glCheckErrors();
}

//-----------------------------------------------------------------------------------
void NvParticlesParticleShape::drawView( M3dView & view, const MDagPath & path, M3dView::DisplayStyle style, M3dView::DisplayStatus displaystatus )
{
	if (!particleContainer)
		return;

    int width = view.portWidth();
    int height = view.portHeight();

    MDagPath thisCameraView;
    view.getCamera( thisCameraView );
    MFnCamera fnCam(thisCameraView);
    double horizontalFOV, verticalFOV;
    fnCam.getPortFieldOfView(width, height, horizontalFOV, verticalFOV);

    MMatrix modelViewMat;
    view.modelViewMatrix(modelViewMat);

    // main rendering happens in here.
    glDraw( path, width, height, verticalFOV, modelViewMat, true );

}
