/* ---------------------------------------------------------------------------
 * This software is in the public domain, furnished "as is", without technical
 * support, and with no warranty, express or implied, as to its usefulness for
 * any purpose.

 * Author: Wil Braithwaite.
 *
 */

#ifndef MAYAUTILS_H_INCLUDED
#define MAYAUTILS_H_INCLUDED

#include <maya/MRenderUtil.h>
#include <maya/MFloatVector.h>
#include <maya/MFnDependencyNode.h>
#include <maya/MString.h>
#include <maya/MGlobal.h>
#include <maya/MFnEnumAttribute.h>
#include <set>

namespace Easy
{

//-----------------------------------------------------------------------------------
#ifndef STDERR_DEFINED
#define STDERR_DEFINED
#define STDERRSEP()      std::cerr << "---------------------------" << std::endl
#define STDERR(a)        std::cerr << #a ": "<<(a)<<std::endl
#define STDERR2(a,b)     std::cerr << #a ": "<<(a)<<"    "<< #b ": "<<(b)<<std::endl
#define STDERR3(a,b,c)   std::cerr << #a ": "<<(a)<<"    "<< #b ": "<<(b)<<"    "<< #c ": "<<(c)<<std::endl
#define STDERR4(a,b,c,d) std::cerr << #a ": "<<(a)<<"    "<< #b ": "<<(b)<<"    "<< #c ": "<<(c)<<"    "<< #d ": "<<(d)<<std::endl
#define STDERR5(a,b,c,d,e) std::cerr << #a ": "<<(a)<<"    "<< #b ": "<<(b)<<"    "<< #c ": "<<(c)<<"    "<< #d ": "<<(d)<<"    "<< #e ": "<<(e)<<std::endl
#define STDL() std::cerr << std::string(__FILE__) << ":" << int(__LINE__) << std::endl
#endif

//-----------------------------------------------------------------------------------
inline MStatus getFloat3Plug(MPlug plug, float* value)
{
    MStatus status = MS::kSuccess;
    MObject obj;
    CHECK_MSTATUS_AND_RETURN_IT( plug.getValue( obj ) );
    MFnNumericData data( obj, &status );
    CHECK_MSTATUS_AND_RETURN_IT( status );
    CHECK_MSTATUS_AND_RETURN_IT( data.getData(value[0], value[1], value[2]) );
    return status;
}

//-----------------------------------------------------------------------------------
inline MFloatVector getFloat3Attr(const MFnDependencyNode &n, const MString& s)
{
    MStatus status = MS::kSuccess;
    MPlug plug = n.findPlug(s, &status);
    CHECK_MSTATUS(status);
    MObject obj;
    CHECK_MSTATUS( plug.getValue( obj ) );
    MFnNumericData data( obj, &status );
    CHECK_MSTATUS( status );
    MFloatVector value;
    CHECK_MSTATUS( data.getData(value[0], value[1], value[2]) );
    return value;
}

//-----------------------------------------------------------------------------------
inline MFloatVector getFloat3Attr(MDataBlock& block, const MObject& attr)
{
    MStatus status = MS::kSuccess;
    MDataHandle inputData = block.inputValue ( attr, &status );
    CHECK_MSTATUS(status);

    MFloatVector value = inputData.asFloatVector();
    /*MFnNumericData data( inputData.data(), &status );
    CHECK_MSTATUS( status );
    CHECK_MSTATUS( data.getData(value[0], value[1], value[2]) );
    */
    return value;
}

//-----------------------------------------------------------------------------------
inline float getFloatAttr(const MFnDependencyNode &n, const MString& s)
{
    MStatus status = MS::kSuccess;
    MPlug plug = n.findPlug(s, &status);
    CHECK_MSTATUS(status);
    float r = plug.asFloat(MDGContext::fsNormal, &status);
    CHECK_MSTATUS(status);
    return r;
}

//-----------------------------------------------------------------------------------
inline float getFloatAttr(MDataBlock& block, const MObject& attr)
{
    MStatus status = MS::kSuccess;
	MDataHandle inputData = block.inputValue ( attr, &status );
    CHECK_MSTATUS(status);
    return inputData.asFloat();
}

//-----------------------------------------------------------------------------------
inline bool getBoolAttr(const MFnDependencyNode &n, const MString& s)
{
    MStatus status = MS::kSuccess;
    MPlug plug = n.findPlug(s, &status);
    CHECK_MSTATUS(status);
    bool r = plug.asBool(MDGContext::fsNormal, &status);
    CHECK_MSTATUS(status);
    return r;
}

//-----------------------------------------------------------------------------------
inline bool getBoolAttr(MDataBlock& block, const MObject& attr)
{
    MStatus status = MS::kSuccess;
	MDataHandle inputData = block.inputValue ( attr, &status );
    CHECK_MSTATUS(status);
    return inputData.asBool();
}

//-----------------------------------------------------------------------------------
inline int getIntAttr(const MFnDependencyNode &n, const MString& s)
{
    MStatus status = MS::kSuccess;
    MPlug plug = n.findPlug(s, &status);
    CHECK_MSTATUS(status);
    int r = plug.asInt(MDGContext::fsNormal, &status);
    CHECK_MSTATUS(status);
    return r;
}

//-----------------------------------------------------------------------------------
inline MObject getObjectAttr(const MFnDependencyNode &n, const MString& s)
{
    MStatus status = MS::kSuccess;
    MPlug plug = n.findPlug(s, &status);
    CHECK_MSTATUS(status);
    MObject r = plug.asMObject(MDGContext::fsNormal, &status);
    CHECK_MSTATUS(status);
    return r;
}

//-----------------------------------------------------------------------------------
inline short getShortAttr(const MFnDependencyNode &n, const MString& s)
{
    MStatus status = MS::kSuccess;
    MPlug plug = n.findPlug(s, &status);
    CHECK_MSTATUS(status);
    short r = plug.asShort(MDGContext::fsNormal, &status);
    CHECK_MSTATUS(status);
    return r;
}

//-----------------------------------------------------------------------------------
inline MString getEnumAttr(const MFnDependencyNode &n, const MString& s)
{
    MStatus status = MS::kSuccess;
    MPlug plug = n.findPlug(s, &status);
    CHECK_MSTATUS(status);
    short v = plug.asShort(MDGContext::fsNormal, &status);
    CHECK_MSTATUS(status);
    MFnEnumAttribute eAttrFn(plug.attribute(), &status);
    CHECK_MSTATUS(status);
    MString x = eAttrFn.fieldName(v, &status);
    CHECK_MSTATUS(status);
    return x;
}

//-----------------------------------------------------------------------------------
inline int getIntAttr(MDataBlock& block, const MObject& attr)
{
    MStatus status = MS::kSuccess;
	MDataHandle inputData = block.inputValue ( attr, &status );
    CHECK_MSTATUS(status);
    return inputData.asInt();
}

//-----------------------------------------------------------------------------------
inline short getShortAttr(MDataBlock& block, const MObject& attr)
{
    MStatus status = MS::kSuccess;
	MDataHandle inputData = block.inputValue ( attr, &status );
    CHECK_MSTATUS(status);
    return inputData.asShort();
}

//-----------------------------------------------------------------------------------
inline MString getEnumAttr(MDataBlock& block, const MObject& attr)
{
    short v = getShortAttr(block, attr);
    MStatus status = MS::kSuccess;
    MFnEnumAttribute eAttrFn(attr, &status);
    CHECK_MSTATUS(status);
    MString s = eAttrFn.fieldName(v, &status);
    CHECK_MSTATUS(status);
    return s;
}

//-----------------------------------------------------------------------------------
inline MString getStringAttr(const MFnDependencyNode &n, const MString& s)
{
    MStatus status = MS::kSuccess;
    MPlug plug = n.findPlug(s, &status);
    CHECK_MSTATUS(status);
    MString r = plug.asString(MDGContext::fsNormal, &status);
    CHECK_MSTATUS(status);
    return r;
}

//-----------------------------------------------------------------------------------
inline MString getStringAttr(MDataBlock& block, const MObject& attr)
{
    MStatus status = MS::kSuccess;
	MDataHandle inputData = block.inputValue ( attr, &status );
    CHECK_MSTATUS(status);
    return inputData.asString();
}

//-----------------------------------------------------------------------------------
inline MString getFileNamePlug(const MPlug &plug)
{
    MStatus status = MS::kSuccess;
    MString r = plug.asString(MDGContext::fsNormal, &status);
    CHECK_MSTATUS(status);

	MString exactName;
	MRenderUtil::exactFileTextureName (r, false, "", exactName, &status);
	CHECK_MSTATUS(status);
    return exactName;
}

//-----------------------------------------------------------------------------------
inline MString getFileNameAttr(const MFnDependencyNode &n, const MString& s)
{
    MStatus status = MS::kSuccess;
    MPlug plug = n.findPlug(s, &status);
	CHECK_MSTATUS(status);
	return getFileNamePlug(plug);
}

//-----------------------------------------------------------------------------------
inline MString getFileNameAttr(MDataBlock& block, const MObject& attr)
{
    MStatus status = MS::kSuccess;
	MDataHandle inputData = block.inputValue ( attr, &status );
    CHECK_MSTATUS(status);
	MString exactName("");
	MRenderUtil::exactFileTextureName (inputData.asString(), false, "", exactName, &status);
	CHECK_MSTATUS(status);
    return exactName;
}

//-----------------------------------------------------------------------------------
static std::set<std::string> usedMessageList;
static void MessageOnce(const MString& s)
{
	//std::set<std::string>::const_iterator it = usedMessageList.begin();
	if(usedMessageList.find(s.asChar()) == usedMessageList.end())
	{
		usedMessageList.insert(s.asChar());
		MGlobal::displayError(s);
	}
}

//-----------------------------------------------------------------------------------
}

#endif // MAYAUTILS_H_INCLUDED
