/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#ifndef NVPARTICLES_COLLISION_MATH_INLINE_H_INCLUDED
#define NVPARTICLES_COLLISION_MATH_INLINE_H_INCLUDED

namespace Collision
{

#define NVPARTICLES_UNITBOXLEN 1
#define NVPARTICLES_UNITSPHERELEN 1
#define NVPARTICLES_EPSILON 0.0000001f

//------------------------------------------------------------------------------------------
// given segment ab and point c, computes closest point d on ab
// also returns t for the position of d, d(t) = a + t(b-a)
inline static NVPARTICLES_CUDA_EXPORT
vec3f closestPointOnSegment(vec3f c, vec3f a, vec3f b, float &t)
{
    vec3f ab = b - a;
    t = dot(c - a, ab) / dot(ab, ab);
    t = clamp(t, 0.0f, 1.0f);
    return a + t * ab;
}

//------------------------------------------------------------------------------------------
/// plane
inline static NVPARTICLES_CUDA_EXPORT
float PlaneExt(vec3f Pw, vec3f V, float _collisionDist, mat44f xform, mat44f xformInv, vec3f extents, vec3f &contactPoint, vec3f &contactNormal)
{
    float penetration = 0;
    const float collisionDist = 0;

    vec3f Pb = xformInv.multiplyPoint(Pw);

    //if ((Pb.x >= -extents.x) && (Pb.x <= extents.x) && (Pb.y >= -extents.y) && (Pb.y <= extents.y))
    {
        // inside range.

        //V -= (prevPb - Pb);
        V = xformInv.multiply(V);

        // BUT we must be certain we were heading towards the exterior of the plane (so we need V).
        if (Pb.z < 0)// && (Pb.z-V.z) >= 0)
        {
            contactNormal = make_vec3f(0, 0, 1);
            contactPoint = (make_vec3f(Pb.x, Pb.y, 0) + contactNormal*collisionDist);

            contactPoint = xform.multiplyPoint(contactPoint);
            contactNormal = xformInv.multiplyTranspose(contactNormal).normalized();

            penetration = length(contactPoint - Pw);
        }
    }

    return penetration;
}

//------------------------------------------------------------------------------------------
/// Evaluate a collision with a halfspace (0,0,1) and get the penetration.
/// (Note that Ncontact is simply the halfspace normal.)
///
/// @param	Pw				world-space point to test.
/// @param	collisionDist	world-space point-radius.
/// @param	xform			cube-space to world-space transform.
/// @param	xformInv		world-space to cube-space transform.
/// @param	Pcontact		[out] world-space contact-point.
///
/// @return					world-space penetration distance. (negative means outside object).
///
inline static NVPARTICLES_CUDA_EXPORT
float UnitHalfSpace(vec3f Pw, float collisionDist, mat44f xform, mat44f xformInv, vec3f &Pcontact, vec3f &Ncontact)
{
    float penetration = 0;

	vec3f Pb = xformInv.multiplyPoint(Pw);
	vec3f collisionRadius = xformInv.multiply(make_vec3f(collisionDist));

    penetration = collisionRadius.z - Pb.z;

    Pcontact = make_vec3f(Pb.x, Pb.y, 0);

    Pcontact = xform.multiplyPoint(Pcontact);
	Ncontact = normalize(make_vec3f(xform.Z()));

    return penetration;
}

//------------------------------------------------------------------------------------------
/// Evaluate a collision with a halfspace and get the penetration.
///
/// @param	Pw				world-space point to test.
/// @param	collisionDist	world-space point-radius.
/// @param	halfspace		(nx,ny,nz,d) where n is normalized, and d is the distance to the origin.
/// @param	Pcontact		[out] world-space contact-point.
/// @param	Ncontact		[out] world-space contact-normal.
///
/// @return					world-space penetration distance. (negative means outside object).
///
inline static NVPARTICLES_CUDA_EXPORT
float HalfSpace(vec3f Pw, float collisionDist, vec4f halfspace, vec3f &Pcontact, vec3f &Ncontact)
{
	Ncontact = make_vec3f(halfspace.x, halfspace.y, halfspace.z);

	vec3f relPos = Pw + (Ncontact * halfspace.w);

	float distance = -dot(relPos, Ncontact);

	Pcontact = Pw - Ncontact * distance;

    return collisionDist + distance;
}

//------------------------------------------------------------------------------------------
/// Evaluate a collision with the interior walls of a unit-cube (-1, +1).
///
/// @param	Pw				world-space point to test.
/// @param	collisionDist	world-space collision radius.
/// @param	xform			cube-space to world-space transform.
/// @param	xformInv		world-space to cube-space transform.
/// @param	contactPoint	[out] world-space contact-point.
/// @param	contactNormal	[out] world-space contact-normal.
///
/// @return					world-space penetration distance. (negative means outside object).
///
inline static NVPARTICLES_CUDA_EXPORT
float UnitBoxInt(const vec3f Pw, const float collisionDist, const mat44f& xform, const mat44f& xformInv, vec3f& contactPoint, vec3f& contactNormal)
{
	// transform point into primitive-space.
	vec3f P = xformInv.multiplyPoint(Pw);
	contactNormal = make_vec3f(0);
	contactPoint = P;

	// TODO: transform collision-distance into primitive-space. (creating a non-uniform scaled sphere)
	vec3f collisionRadius = xformInv.multiply(make_vec3f(collisionDist));

	float xlen = length(xform.X());
	float ylen = length(xform.Y());
	float zlen = length(xform.Z());

	vec3f Pcontact[6];
	vec3f Ncontact[6];
	float penetration[6];

	penetration[0] = HalfSpace(P, collisionRadius.x, make_vec4f(-1,0,0,1), Pcontact[0], Ncontact[0]) * xlen;
	penetration[1] = HalfSpace(P, collisionRadius.x, make_vec4f(1,0,0,1), Pcontact[1], Ncontact[1]) * xlen;
	penetration[2] = HalfSpace(P, collisionRadius.y, make_vec4f(0,-1,0,1), Pcontact[2], Ncontact[2]) * ylen;
	penetration[3] = HalfSpace(P, collisionRadius.y, make_vec4f(0,1,0,1), Pcontact[3], Ncontact[3]) * ylen;
	penetration[4] = HalfSpace(P, collisionRadius.z, make_vec4f(0,0,-1,1), Pcontact[4], Ncontact[4]) * zlen;
	penetration[5] = HalfSpace(P, collisionRadius.z, make_vec4f(0,0,1,1), Pcontact[5], Ncontact[5]) * zlen;

	int maxIndex = 0;
	float maxPenetration = 0;
	for(int i=0;i<6;++i)
	{
		if (penetration[i] > maxPenetration)
		{
			maxPenetration = penetration[i];
			maxIndex = i;
		}
	}

    //contactPoint = xform.multiplyPoint(contactPoint);
	// flip the normal!
    contactNormal = normalize(xformInv.multiplyTranspose(Ncontact[maxIndex]));

    // get penetration in world-space!
	//if (maxPenetration > 0)
	//	maxPenetration = length(contactPoint-Pw);

	return maxPenetration;
}

//------------------------------------------------------------------------------------------
/// collide with a unit sphere's interior
///
/// @param	Pw				world-space point to test.
/// @param	collisionDist	world-space collision radius.
/// @param	xform			cube-space to world-space transform.
/// @param	xformInv		world-space to cube-space transform.
/// @param	contactPoint	[out] world-space contact-point.
/// @param	contactNormal	[out] world-space contact-normal.
///
/// @return					world-space penetration distance. (negative means outside object).
///
inline static NVPARTICLES_CUDA_EXPORT
float UnitSphereInt(vec3f Pw, float _collisionDist, mat44f xform, mat44f xformInv, vec3f &contactPoint, vec3f &contactNormal)
{
    const float collisionDist = 0;
    const float bl = NVPARTICLES_UNITSPHERELEN;
    float penetration = 0;

    vec3f p = xformInv.multiplyPoint(Pw);

    //nb. collisionDist is already been scaled.
    float dist = length(p);
    contactNormal = -(p / dist);

    penetration = collisionDist + (dist - bl);

    contactPoint = p + (penetration*contactNormal);

	contactPoint = xform.multiplyPoint(contactPoint);
    contactNormal = xformInv.multiplyTranspose(contactNormal).normalized();

    return penetration;
}

//------------------------------------------------------------------------------------------
/// collide with a unit sphere's exterior
///
/// @param	Pw				world-space point to test.
/// @param	collisionDist	world-space collision radius.
/// @param	xform			cube-space to world-space transform.
/// @param	xformInv		world-space to cube-space transform.
/// @param	contactPoint	[out] world-space contact-point.
/// @param	contactNormal	[out] world-space contact-normal.
///
/// @return					world-space penetration distance. (negative means outside object).
///
inline static NVPARTICLES_CUDA_EXPORT
float UnitSphereExt(vec3f Pw, float _collisionDist, mat44f xform, mat44f xformInv, vec3f &contactPoint, vec3f &contactNormal)
{
    const float collisionDist = 0;
    const float radius = NVPARTICLES_UNITSPHERELEN;
    float penetration = 0;

    vec3f p = xformInv.multiplyPoint(Pw);

    //nb. collisionDist is already been scaled.
    float dist = length(p);
    contactNormal = (p / dist);

    penetration = collisionDist + (radius - dist);

    contactPoint = p + (penetration*contactNormal);

    contactPoint = xform.multiplyPoint(contactPoint);
    contactNormal = xformInv.multiplyTranspose(contactNormal).normalized();

    // get penetration in world-space
    //penetration = ((penetration>0)?1:-1) * length(contactPoint-Pw);
    //penetration = collisionDist + (dist - bl);
    ///penetration = -(collisionDist + (dist - bl));

    return penetration;
}

//------------------------------------------------------------------------------------------
/// collide with a unit capsule's exterior
///
/// @param	Pw				world-space point to test.
/// @param	collisionDist	world-space collision radius.
/// @param	xform			cube-space to world-space transform.
/// @param	xformInv		world-space to cube-space transform.
/// @param	contactPoint	[out] world-space contact-point.
/// @param	contactNormal	[out] world-space contact-normal.
///
/// @return					world-space penetration distance. (negative means outside object).
///
inline static NVPARTICLES_CUDA_EXPORT float
UnitCapsuleExt(const vec3f Pw, const float _collisionDist, const float len, const mat44f xform, const mat44f xformInv, vec3f &contactPoint, vec3f &contactNormal)
{
    const float collisionDist = 0;
    const float radius = NVPARTICLES_UNITSPHERELEN;
    float penetration = 0;
    float halflen = len/2.f;
	float t;
	vec3f nearestPt;

    vec3f p = xformInv.multiplyPoint(Pw);

    nearestPt = closestPointOnSegment(p, make_vec3f(0,0,-halflen), make_vec3f(0,0,halflen), t);

    vec3f relPos = p - nearestPt;
    float dist = length(relPos);
    contactNormal = relPos/dist;

    // get amount we have penetrated (negative if not)
    penetration = collisionDist + (radius - dist);

    contactPoint = p + contactNormal*penetration;

	contactPoint = xform.multiplyPoint(contactPoint);
    contactNormal = xformInv.multiplyTranspose(contactNormal).normalized();

    return penetration;
}

//------------------------------------------------------------------------------------------
/// collide with a convex-hull (exterior)
///
/// @param	Pw				world-space point to test.
/// @param	collisionDist	world-space collision radius.
/// @param	xform			cube-space to world-space transform.
/// @param	xformInv		world-space to cube-space transform.
/// @param	numPlanes		number of convex planes.
/// @param	planes			Array of planes that fomr the convex-hull.
/// @param	contactPoint	[out] world-space contact-point.
/// @param	contactNormal	[out] world-space contact-normal.
///
/// @return					world-space penetration distance. (negative means outside object).
///
inline static NVPARTICLES_CUDA_EXPORT
float ConvexPlanesExt(vec3f Pw, float _collisionDist, mat44f xform, mat44f xformInv, const int numPlanes, const vec4f* planes, vec3f& contactPoint, vec3f& contactNormal)
{
    const float collisionDist = 0;
    float penetration = 0;

    vec3f p = xformInv.multiplyPoint(Pw);

    int isOutside = 0;
    float closestDist = -1e10;
    vec3f closestNormal;

    // test each plane...
    for (int i=0; i<numPlanes; i++)
	{
        const vec4f& plane = planes[i];
        float dist = dot(p, make_vec3f(plane.x,plane.y,plane.z)) - plane.w;

        if (dist < collisionDist)
		{
            // inside plane...
            if (dist > closestDist)
			{
                closestDist = dist;
                closestNormal = make_vec3f(plane.x,plane.y,plane.z);
            }
        }
		else
		{
            // outside
            isOutside = 1;
            break;
        }
    }

    if (!isOutside)
	{
        // point is inside all planes...
		penetration = collisionDist + closestDist;

        contactPoint = p + penetration*closestNormal;
		contactPoint = xform.multiplyPoint(contactPoint);
		contactNormal = xformInv.multiplyTranspose(contactNormal).normalized();
    }

    return penetration;
}

//------------------------------------------------------------------------------------------
inline static NVPARTICLES_CUDA_EXPORT
float UnitBoxExt(vec3f p, float _collisionDist, mat44f xform, mat44f xformInv, vec3f &contactPoint, vec3f &contactNormal, float scale=1.f)
{
    float penetration = 0;
    const float collisionDist = 0;
    const float bl = NVPARTICLES_UNITBOXLEN;

    p = make_vec3f(xformInv * make_vec4f(p.x,p.y,p.z,1));

    // this MUST return true if p == low or high bounds (to avoid div by zero in other condition)
    if (((p.x >= -bl) && (p.x <= bl) && (p.y >= -bl) && (p.y <= bl) && (p.z >= -bl) && (p.z <= bl)))
    {
        // find closest plane
        float minDist = 1e10f;
        float d;
        d = bl - p.x;
        if (d < minDist)
        {
            minDist = d;
            contactNormal = make_vec3f(1, 0, 0);
        }
        d = bl - p.y;
        if (d < minDist)
        {
            minDist = d;
            contactNormal = make_vec3f(0, 1, 0);
        }
        d = bl - p.z;
        if (d < minDist)
        {
            minDist = d;
            contactNormal = make_vec3f(0, 0, 1);
        }

        d = p.x - -bl;
        if (d < minDist)
        {
            minDist = d;
            contactNormal = make_vec3f(-1, 0, 0);
        }
        d = p.y - -bl;
        if (d < minDist)
        {
            minDist = d;
            contactNormal = make_vec3f(0, -1, 0);
        }
        d = p.z - -bl;
        if (d < minDist)
        {
            minDist = d;
            contactNormal = make_vec3f(0, 0, -1);
        }

        // get amount we have penetrated (negative if not)
        penetration = (minDist + collisionDist);
        contactPoint = (p + contactNormal*penetration);// / scale;

        contactPoint = xform.multiplyPoint(contactPoint);
        contactNormal = xformInv.multiplyTranspose(contactNormal).normalized();
    }
    else if(collisionDist != 0)
    {
        // find closest point on unit box
        // (this only works if point is outside box!)
        vec3f q = fmaxf(fminf(p, make_vec3f(bl)), make_vec3f(-bl));

        vec3f d = p - q;
        float dist = length(d);
        contactNormal = d / dist;   // not really correct on corners?

        // project outside of box
        contactPoint = (q + contactNormal*collisionDist);// / scale;

        contactPoint = xform.multiplyPoint(contactPoint);
        contactNormal = xformInv.multiplyTranspose(contactNormal).normalized();

        penetration = -(dist - collisionDist);
    }

    // return penetration (negative if it is outside the box)
    return penetration;
}


//------------------------------------------------------------------------------------------
// test if point is inside box
inline static NVPARTICLES_CUDA_EXPORT
bool pointInsideBox(vec3f p, vec3f low, vec3f high)
{
    return ((p.x >= low.x) && (p.x <= high.x) &&
            (p.y >= low.y) && (p.y <= high.y) &&
            (p.z >= low.z) && (p.z <= high.z));
}

//------------------------------------------------------------------------------------------
// find closest point on box to p
// by clamping each coordinate to box min/max
// only works if point is outside box!
inline static NVPARTICLES_CUDA_EXPORT
vec3f closestPointOnBox(vec3f p, vec3f low, vec3f high)
{
    return fmaxf(fminf(p, high), low);
}

//------------------------------------------------------------------------------------------
inline static NVPARTICLES_CUDA_EXPORT
float collideBoxExt(vec3f p, float collisionDist, vec3f low, vec3f high, mat44f xform, mat44f xformInv, vec3f &contactPoint, vec3f &contactNormal)
{
    p = make_vec3f(xformInv * make_vec4f(p.x,p.y,p.z,1));

    // this MUST return true if p == low or high bounds
    // (to avoid div by zero in other condition)
    if (pointInsideBox(p, low, high))
    {
        // find closest plane
        float minDist = 1e10f;
        float d;
        d = high.x - p.x;
        if (d < minDist)
        {
            minDist = d;
            contactNormal = make_vec3f(1, 0, 0);
        }
        d = high.y - p.y;
        if (d < minDist)
        {
            minDist = d;
            contactNormal = make_vec3f(0, 1, 0);
        }
        d = high.z - p.z;
        if (d < minDist)
        {
            minDist = d;
            contactNormal = make_vec3f(0, 0, 1);
        }

        d = p.x - low.x;
        if (d < minDist)
        {
            minDist = d;
            contactNormal = make_vec3f(-1, 0, 0);
        }
        d = p.y - low.y;
        if (d < minDist)
        {
            minDist = d;
            contactNormal = make_vec3f(0, -1, 0);
        }
        d = p.z - low.z;
        if (d < minDist)
        {
            minDist = d;
            contactNormal = make_vec3f(0, 0, -1);
        }

        // get amount we have penetrated (negative if not)
        float penetration = (minDist + collisionDist);
        contactPoint = p + contactNormal*penetration;

        contactPoint = xform.multiplyPoint(contactPoint);
        contactNormal = xformInv.multiplyTranspose(contactNormal).normalized();


        return -penetration;
    }
    else
    {
        // find closest point on box
        vec3f q = closestPointOnBox(p, low, high);

        vec3f d = p - q;
        float dist = length(d);
        contactNormal = d / dist;   // not really correct on corners?

        // project outside of box
        contactPoint = q + contactNormal*collisionDist;

        contactPoint = xform.multiplyPoint(contactPoint);
        contactNormal = xformInv.multiplyTranspose(contactNormal).normalized();

        // return penetration (negative if it is inside the box)
        return dist - collisionDist;
    }
}

} // end of namespace

#endif // NVPARTICLES_COLLISION_MATH_INLINE_H_INCLUDED
