/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#ifndef NVPARTICLES_COLOR_MATH_CUDA_INLINE_H_INCLUDED
#define NVPARTICLES_COLOR_MATH_CUDA_INLINE_H_INCLUDED

//------------------------------------------------------------------------------------------
enum COLOR_GRADIENT
{
	COLOR_GRADIENT_NONE,
	COLOR_GRADIENT_WHITE_TO_BLACK,
    COLOR_GRADIENT_BLACK_TO_WHITE,
	COLOR_GRADIENT_BLUE_TO_RED_HSV,
	COLOR_GRADIENT_TEMPERATURE,
};

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT
float4 computeColorGradient(int colorGradient, float value)
{
	float4 color = make_float4(0,0,0,0);
	switch (colorGradient)
	{
    case COLOR_GRADIENT_BLACK_TO_WHITE:
        color = make_float4(value,value,value,1);
        break;

    case COLOR_GRADIENT_WHITE_TO_BLACK:
        color = make_float4(1-value,1-value,1-value,1);
        break;

	case COLOR_GRADIENT_BLUE_TO_RED_HSV:
		{
			float h = clamp((1-value)*245.0f,0.0f,245.0f);
			vec3f rgb = hsvToRgb(h,0.5f,1);
			color = make_float4(rgb.x,rgb.y,rgb.z,1);
		}
		break;

	case COLOR_GRADIENT_TEMPERATURE:
		{
			float v = value;
			vec3f rgb = pseudoTemperature(v);
			color = make_float4(rgb.x,rgb.y,rgb.z,1);
		}
		break;
	}
	return color;
}

//------------------------------------------------------------------------------------------
// a simple color palette for diagnostic coloring...
//
#define NVPARTICLES_COLOR_PALETTE_SIZE 10
#ifdef __CUDACC__
__constant__ float4 d_colorPalette[NVPARTICLES_COLOR_PALETTE_SIZE] = {{1,0,0,1}, {0,1,0,1}, {0,0,1,1}, {1,1,0,1}, {1,0,1,1}, {0,1,1,1}, {1,1,1,1}, {0.5,0.5,0.5,1}, {0.5,1,0,1}, {0.5,0,1,1}};
#endif
static const float4 h_colorPalette[NVPARTICLES_COLOR_PALETTE_SIZE] = {{1,0,0,1}, {0,1,0,1}, {0,0,1,1}, {1,1,0,1}, {1,0,1,1}, {0,1,1,1}, {1,1,1,1}, {0.5,0.5,0.5,1}, {0.5,1,0,1}, {0.5,0,1,1}};
#ifdef __CUDA_ARCH__
#define NVPARTICLES_PALETTE_COLOR(i) d_colorPalette[i%NVPARTICLES_COLOR_PALETTE_SIZE]
#else
#define NVPARTICLES_PALETTE_COLOR(i) h_colorPalette[i%NVPARTICLES_COLOR_PALETTE_SIZE]
#endif

//-----------------------------------------------------------------------------------
#endif // NVPARTICLES_COLOR_MATH_CUDA_INLINE_H_INCLUDED
