/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#include "NvParticlesForces.h"
#include "NvParticlesForcesData.h"
#include "cuda_utils.h"
#include "std_utils.h"
#include "NvParticlesProfiler.h"

namespace Easy
{
namespace NvParticles
{

int ParticleForces::debugLevel = 0;

ForceData ParticleForces::Invalid;

//------------------------------------------------------------------------------------------
ParticleForces::ParticleForces()
{
    clear();
    forces.globalDamping = 0.99;
    forces.particleMass = 1.0;
    forces.internalScale = 1.0;
    forces.scale = 1.0;
}

//------------------------------------------------------------------------------------------
ForceData& ParticleForces::get(const std::string& name)
{
    for (int i=0;i<nextForceIndex;++i)
    {
        if (names[i] == name)
            return forces.forces[i];
    }
    return Invalid;
}

//------------------------------------------------------------------------------------------
bool ParticleForces::remove(const std::string& name)
{
    ForceData& data = get(name);
    if (data.type == ForceData::FORCE_NONE)
        return false;

    if (debugLevel > 0)
        printInfo("Removing force: " + name);

    memset((void*)&data, 0, sizeof(ForceData));

    return true;
}

//------------------------------------------------------------------------------------------
bool ParticleForces::add(const std::string& name, ForceData f)
{
    bool rc = true;

    if (nextForceIndex == MAX_FORCES)
    {
        // compact the list of forces!
        for (int i=0;i<nextForceIndex;++i)
        {
            if (forces.forces[i].type != ForceData::FORCE_NONE)
                continue;
            --nextForceIndex;
            forces.forces[i] = forces.forces[nextForceIndex];
            forces.forces[nextForceIndex].type = ForceData::FORCE_NONE;
        }
    }

    int useIndex = nextForceIndex;
    for (int i=0;i<nextForceIndex;++i)
    {
        if (names[i] == name)
        {
            rc = false;
            useIndex = i;

            if (debugLevel > 0)
                printInfo("Updating force: " + name);
        }
    }

    if (useIndex < MAX_FORCES)
    {
        forces.forces[useIndex] = f;
        names[useIndex] = name;
        needUpload = true;

        if (useIndex == nextForceIndex)
        {
            if (debugLevel > 0)
                printInfo("Created force: " + name);

            nextForceIndex++;
        }
    }
    else
    {
        printError(Stringf("Unable to create force: %s (more than %d forces)", name.c_str(), MAX_FORCES));
        rc = false;
    }

    return rc;
}

//------------------------------------------------------------------------------------------
void ParticleForces::clear()
{
    for (int i=0;i<MAX_FORCES;++i)
        forces.forces[i].type = ForceData::FORCE_NONE;
    nextForceIndex = 0;
    needUpload = true;
}

//------------------------------------------------------------------------------------------
bool ParticleForces::add(const std::string& name, int type, float* origin, float* axis, float magnitude, float maxDistance, float attenuation)
{
    ForceData f;
    f.type = type;

	f.origin = make_vec3f(0);
	if (origin)
	    f.origin = make_vec3f(origin[0], origin[1], origin[2]);

	f.axis = make_vec3f(0);
	if (axis)
		f.axis = make_vec3f(axis[0], axis[1], axis[2]);

    f.magnitude = magnitude;
    f.maxDistance = (maxDistance < 0)?9999999:maxDistance;
    f.attenuation = std::max(0.0f, attenuation);

    return add(name,f);
}

//------------------------------------------------------------------------------------------
}
}
