/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#include "NvParticlesIteratorCudaInline.h"
#include "NvParticlesForcesMathInline.h"
#include "math_utils_cuda.h"

//------------------------------------------------------------------------------------------
#ifdef __CUDA_ARCH__
#define NVPARTICLES_FORCE_PARAM(x) d_forcesParams.x
#else
#define NVPARTICLES_FORCE_PARAM(x) h_forcesParams.x
#endif

__device__ __constant__ ForcesData d_forcesParams;
static ForcesData h_forcesParams;

#ifdef NVPARTICLES_USE_FORCE_DATA_TEXTURE
// not yet implemented...
texture<ForceData, 1, cudaReadModeElementType> d_forcesDataTexture;
#endif

//------------------------------------------------------------------------------------------
//! update constants
//!
extern "C" inline static
void uploadForces(const ForcesData& params, cudaStream_t stream=0)
{
#ifdef NVPARTICLES_USE_FORCE_DATA_TEXTURE
    h_forcesParams = params;
	NVPARTICLES_CUDA_SAFE_CALL( cudaMemcpyToSymbolAsync(d_forcesParams, &params, sizeof(ForcesData)-(sizeof(ForceData)*MAX_FORCES), 0, cudaMemcpyHostToDevice, stream ));
    // not yet implemented...
	cudaChannelFormatDesc channelDesc = cudaCreateChannelDesc(MAX_FORCES, 0, 0, 0, cudaChannelFormatKindNone);
    NVPARTICLES_CUDA_SAFE_CALL(cudaBindTexture(0, d_forcesDataTexture, d_imageData, channelDesc, (sizeof(ForceData)*MAX_FORCES)));
#else
    h_forcesParams = params;
    NVPARTICLES_CUDA_SAFE_CALL( cudaMemcpyToSymbolAsync(d_forcesParams, &params, sizeof(ForcesData), 0, cudaMemcpyHostToDevice, stream ));
#endif
}

//------------------------------------------------------------------------------------------
//! evaluate an external force.
//!
NVPARTICLES_CUDA_EXPORT static
float3 evaluateForce(const ForceData& c, float3 P, float3 V)
{
    // The positional units of forces is NOT the same as the magnitudes,
    // which are specified in world units such as (metres/second)
    float unitScaleRatio = NVPARTICLES_FORCE_PARAM(scale) / NVPARTICLES_FORCE_PARAM(internalScale);

    if(c.type == ForceData::FORCE_UNIFORM)
    {
        return Force::UniformFalloff(P, make_float3(c.origin*unitScaleRatio), make_float3(c.axis), c.magnitude, c.maxDistance*unitScaleRatio, c.attenuation);
    }
    else if(c.type == ForceData::FORCE_RADIAL)
    {
        return Force::RadialFalloff(P, make_float3(c.origin*unitScaleRatio), c.magnitude, c.maxDistance*unitScaleRatio, c.attenuation);
    }
    else if(c.type == ForceData::FORCE_VORTEX)
    {
		return Force::VortexFalloff(P, make_float3(c.origin*unitScaleRatio), make_float3(c.axis), c.magnitude, c.maxDistance*unitScaleRatio, c.attenuation);
    }
    else if(c.type == ForceData::FORCE_DRAG)
    {
        return Force::DragFalloff(P, V, make_float3(c.origin*unitScaleRatio), make_float3(c.axis), NVPARTICLES_FORCE_PARAM(deltaTime), c.magnitude, c.maxDistance*unitScaleRatio, c.attenuation);
    }
    else if(c.type == ForceData::FORCE_AIR)
    {
        return Force::AirFalloff(P, V, make_float3(c.origin*unitScaleRatio), make_float3(c.axis), NVPARTICLES_FORCE_PARAM(deltaTime), c.magnitude, c.maxDistance*unitScaleRatio, c.attenuation);
    }
    else if(c.type == ForceData::FORCE_AXIS)
    {
        return Force::AxisFalloff(P, make_float3(c.origin*unitScaleRatio), make_float3(c.axis), c.magnitude, c.maxDistance*unitScaleRatio, c.attenuation);
    }

    return make_float3(0);
}

//------------------------------------------------------------------------------------------
//! Default force iterator.
//!
struct DefaultForceIterator : SimpleIterator
{
    float3 result;
    float3 P;
    float3 V;

    inline static NVPARTICLES_CUDA_EXPORT
    void pre(DefaultForceIterator& it, const uint &i)
    {
        it.result = make_float3(0);
    }

    inline static NVPARTICLES_CUDA_EXPORT
    bool item(DefaultForceIterator& it, const uint &i)
    {
        const ForceData& c = NVPARTICLES_FORCE_PARAM(forces[it.__iteratorIndex]);
        it.result += evaluateForce(c, it.P, it.V);
        return true;
    }
};

//------------------------------------------------------------------------------------------
//! Accumulate all the forces.
//!
template<class Iterator>
inline NVPARTICLES_CUDA_EXPORT
void iterateForces(Iterator& it, const uint &i)
{
    simpleIterate<Iterator, MAX_FORCES>(it, i);
}

//------------------------------------------------------------------------------------------

