#ifndef NVPARTICLES_FORCES_MATH_H
#define NVPARTICLES_FORCES_MATH_H

#include <cutil_math.h>
#include "math_utils.h"

namespace Force
{

//------------------------------------------------------------------------------------------
//! A force in a direction.
//!
inline NVPARTICLES_CUDA_EXPORT float3 Uniform(float3 dir, float magnitude)
{
    return dir * magnitude;
}

//------------------------------------------------------------------------------------------
//! A force in a direction, with falloff.
//!
inline NVPARTICLES_CUDA_EXPORT float3 UniformFalloff(float3 pos, float3 origin, float3 dir, float magnitude, float maxDistance, float attenuation)
{
    float3 relPos = pos - origin;
    float distance = length(relPos);
	float factor = 1.0f / (expf(distance * attenuation));

	if (distance > maxDistance)
		factor = 0;

	return magnitude * dir * factor;
}

//------------------------------------------------------------------------------------------
//! A force which opposes the velocity.
//!
inline NVPARTICLES_CUDA_EXPORT float3 Drag(float3 vel, float3 direction, float dt, float magnitude)
{
	if(direction.x == 0 && direction.y == 0 && direction.z == 0)
		direction = normalize(vel);
	float ufactor = dot(vel, direction) + 0.001f;
    magnitude *= -ufactor;// / (dt*dt);

    return direction * magnitude;
}

//------------------------------------------------------------------------------------------
//! A force which opposes the velocity, with falloff.
//!
inline NVPARTICLES_CUDA_EXPORT float3 DragFalloff(float3 pos, float3 vel, float3 origin, float3 direction, float dt, float magnitude, float maxDistance, float attenuation)
{
	if(direction.x == 0 && direction.y == 0 && direction.z == 0)
		direction = normalize(vel);
	float ufactor = dot(vel, direction) + 0.001f;
    magnitude *= -ufactor;// / (dt*dt);

    float3 relPos = pos - origin;
    float distance = length(relPos);
	float factor = 1.0f / (expf(distance * attenuation));

	if (distance > maxDistance)
		factor = 0;

	return direction * magnitude * factor;
}

//------------------------------------------------------------------------------------------
//! A force in a specific direction, with falloff and spread.
//! magnitude = speed the objects match the airvelocity. (NB. different than Maya)
//!
inline NVPARTICLES_CUDA_EXPORT float3 AirFalloff(float3 pos, float3 vel, float3 origin, float3 direction, float dt, float magnitude, float maxDistance, float attenuation)
{
	// direction has magnitude multiplied in.
	//direction *= magnitude;
	float3 relVel = (direction - vel);

    float3 relPos = pos - origin;
    float distance = length(relPos);
	float factor = 1.0f / (expf(distance * attenuation));
	if (distance > maxDistance)
		factor = 0;

    //float ufactor = dot(relVel, direction);

	return relVel * magnitude * factor;
}

//------------------------------------------------------------------------------------------
//! A force away from a point.
//! Negative values will attract.
//!
inline NVPARTICLES_CUDA_EXPORT float3 Radial(float3 pos, float3 origin, float magnitude)
{
    float3 relPos = pos - origin;
    return relPos * magnitude;
}

//------------------------------------------------------------------------------------------
//! A force away from a point, but with falloff.
//!
inline NVPARTICLES_CUDA_EXPORT float3 RadialFalloff(float3 pos, float3 origin, float magnitude, float maxDistance, float attenuation)
{
    float3 relPos = pos - origin;
    float distance = length(relPos);
	float factor = 1.0f / (expf(distance * attenuation));

	if (distance > maxDistance)
		factor = 0;

	return magnitude * (relPos/distance) * factor;
}

//------------------------------------------------------------------------------------------
//! A force around an axis, through a point.
//! (axis must be normalized)
//!
inline NVPARTICLES_CUDA_EXPORT float3 Vortex(float3 pos, float3 origin, float3 axis, float magnitude)
{
    float3 relPos = pos - origin;
    relPos = normalize(relPos);
    float3 force = cross(relPos, axis);

    return force * magnitude;
}

//------------------------------------------------------------------------------------------
//! A force around an axis, through a point, with falloff.
//! (axis must be normalized)
//!
inline NVPARTICLES_CUDA_EXPORT float3 VortexFalloff(float3 pos, float3 origin, float3 axis, float magnitude, float maxDistance, float attenuation)
{
    float3 relPos = pos - origin;
    float distance = length(relPos);
	float factor = 1.0f / (expf(distance * attenuation));

	if (distance > maxDistance)
		factor = 0;

	float3 force = cross(relPos/distance, axis);
	force = force * factor * magnitude;

    return force;
}

//------------------------------------------------------------------------------------------
inline NVPARTICLES_CUDA_EXPORT float3 AxisFalloff(float3 pos, float3 origin, float3 axis, float magnitude, float maxDistance, float attenuation)
{
    float3 relPos = pos - origin;
    float distance = length(relPos);
	float factor = 1.0f / (expf(distance * attenuation));

	if (distance > maxDistance)
		factor = 0;

	return make_float3(0);
/*
    float innerR = 0.0f;
    float outerR = 1.0f;
    float3 upVector = axis;
    float upStrength = 0.00005f;

    float3 pToAxis = origin - make_float3(P.x, 0.0f, P.z);
    float r = length(pToAxis);

    float distFalloff = 1.0f - smoothstep(innerR, outerR, r);
    float3 tangent = make_float3(pToAxis.z, 0.0f, -pToAxis.x);

    return distFalloff * ((pToAxis * magnitudeCentripetal) + (tangent * magnitudeTangent) + upVector*upStrength);
*/
}

} // end namespace

//------------------------------------------------------------------------------------------
#if 0
// collide two spheres using DEM method
__device__ float3 collideWithSphere(float3 *pP, float3 *pDP,
                                    float3 posB, float3 velB,
                                    float radiusA, float radiusB,
                                    float attraction,
                                    float spring,
                                    float damping,
                                    float shear)
{
    float3 posA = *pP;
    float3 velA = *pDP;

    float3 offset = posA - posB;
    float dist = length (offset);
    float collideDist = radiusA + radiusB;

    float3 force = make_float3(0.0f);
    if (dist < collideDist)
    {
        float3 normOffset = offset / dist;

        // position sphereA on the edge of sphereB
        *pP = (normOffset * collideDist) + posB;

        float3 relVel = velA - velB;

        float vdot = dot (relVel, normOffset);

        if (vdot < 0.0f)
        {
            // relative tangential velocity
            float3 velCorrection = (-vdot * normOffset);
            float3 tanVel = relVel + velCorrection;

            // spring force
            force = spring * (collideDist - dist) * normOffset;
            // dashpot (damping) force
            force += damping * relVel;
            // tangential shear force
            force += shear * tanVel;
            // attraction
            force += attraction * -offset;

            *pDP = force;
        }
    }

    return force;
}


__device__ __host__ void carveWithSphere(float3 *pP, float3 *pDP, float3 Origin, float Radius, float Restituion, float FrictionCooeficient)
{
    float3 Diff;
    float Distance;

    Diff = *pP - Origin;
    Distance = length (Diff);

    if (Distance - Radius > 0.1f)
    {
        float3 NormDiff = Diff * (1.0f / Distance);

        float3 V = *pDP;
        float VDot = dot (V, NormDiff);

        if (VDot < 0.0f)
        {
            // we are outside the sphere.
            *pP = (NormDiff * Radius) + Origin;
            float3 VelocityCorrection = NormDiff * -VDot;
            float3 VTangent = V + VelocityCorrection;
            *pDP = (VTangent * FrictionCooeficient) + (VelocityCorrection * Restituion);
        }
        /*else
        {
            // we are inside the sphere.
            float3 VelocityCorrection = NormDiff * VDot;
            float3 VTangent = V + VelocityCorrection;
            *pDP = (VTangent * FrictionCooeficient) + (VelocityCorrection * Restituion);
        }*/
    }
}


__device__ __host__ void collideWithSphere(float3 *pP, float3 *pDP, float3 Origin, float Radius, float Restituion, float FrictionCooeficient)
{
    float3 Diff;
    float Distance;

    Diff = *pP - Origin;
    Distance = length (Diff);

    if (Distance < Radius)
    {
        float3 NormDiff = Diff * (1.0f / Distance);
        *pP = (NormDiff * Radius) + Origin;

        float3 V = *pDP;
        float VDot = dot (V, NormDiff);

        if (VDot < 0.0f)
        {
            float3 VelocityCorrection = NormDiff * -VDot;
            float3 VTangent = V + VelocityCorrection;
            *pDP = (VTangent * FrictionCooeficient) + (VelocityCorrection * Restituion);
        }
    }
}


__device__ void collideWithPlane(float3 *pP, float3 *pDP, float3 Origin, float3 Axis, float Restituion, float FrictionCooeficient)
{
    float3 Diff;

    Diff = *pP - Origin;
    float Dot = dot (Axis, Diff);

    if (Dot < 0.0f)
    {
        *pP = (Axis * -Dot) + *pP;

        float3 V = *pDP;
        float VDot = dot (V, Axis);

        if (VDot < 0.0f)
        {
            float3 VelocityCorrection = Axis * -VDot;
            float3 VTangent = V + VelocityCorrection;
            *pDP = (VTangent * FrictionCooeficient) + (VelocityCorrection * Restituion);
        }
    }
}

#endif


#endif
