/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#include "NvParticlesManager.h"
#include "NvParticlesProfiler.h"

#include "NvParticlesParticleSolverImpl.h"
#include "NvParticlesParticleRenderer.h"
#include "NvParticlesPrimitives.h"
#include "NvParticlesForces.h"
#include "NvParticlesParticleRendererImpl_Point.h"
#include "NvParticlesParticleRendererImpl_Sphere.h"

//----------------------------------------------------------------------------------------------
template<> Easy::NvParticles::Manager* Easy::Singleton<Easy::NvParticles::Manager>::_singleton = 0;


namespace Easy
{
namespace NvParticles
{

//----------------------------------------------------------------------------------------------
Manager& Manager::getSingleton(void)
{
    assert( _singleton );
    return ( *_singleton );
}

//----------------------------------------------------------------------------------------------
Manager::Manager()
{
    printInfo("Creating NvParticles Manager.");

    new Profiler;

    // these are statically linked...
    rendererFactory.registerType("points", Easy::NvParticles::ParticleRendererImpl_Point::creator);
    rendererFactory.registerType("spheres", Easy::NvParticles::ParticleRendererImpl_Sphere::creator);
}

//----------------------------------------------------------------------------------------------
Manager::~Manager()
{
    printInfo("Destroying NvParticles Manager.");

    rendererFactory.deregisterType("points");
    rendererFactory.deregisterType("spheres");
    solverFactory.deregisterType("wcsph");
}

//----------------------------------------------------------------------------------------------
ParticleRendererImpl* Manager::createRenderer(const std::string& name) const
{
	return static_cast<ParticleRendererImpl*>(rendererFactory.create(name));
}

//----------------------------------------------------------------------------------------------
ParticleSolverImpl* Manager::createSolver(const std::string& name) const
{
	return static_cast<ParticleSolverImpl*>(solverFactory.create(name));
}

//----------------------------------------------------------------------------------------------
ParticleSolverSpecPtr Manager::getParticleSolverSpec(const std::string& name)
{
    ParticleSolverSpecPtr spec;
    ParticleSolverSpecs::const_iterator it = _particleSolverSpecs.find(name);
    if (it != _particleSolverSpecs.end())
        spec = it->second;

    if (!spec)
    {
        // create a temporary solver to get the definition
	    SharedPtr<ParticleSolverImpl> s = createSolver(name);

        /// OPTIMIZE:
        // do this all from static initializers so we don't need to create an instance!

	    if(s)
	    {
		    s->initialize();

		    spec = new ParticleSolverSpec();
		    spec->parameterDefinitions = s->parameterDefinitions();
            spec->bufferSpecs = s->getBufferDefinitions();

            _particleSolverSpecs[name] = spec;
	    }
    }

	return spec;
}

//------------------------------------------------------------------------------------------
}
}
