/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#include "NvParticlesParameters.h"

namespace Easy
{
namespace NvParticles
{

//-----------------------------------------------------------------------------------
ParameterValue::ParameterValue()
{
}

//-----------------------------------------------------------------------------------
ParameterValue::ParameterValue(const ParameterValue& b)
{
	_any = b._any;
}

//-----------------------------------------------------------------------------------
ParameterValue& ParameterValue::operator=(const ParameterValue& b)
{
	_any = b._any;
	return (*this);
}

//-----------------------------------------------------------------------------------
bool ParameterValue::valid() const
{
	return !_any.empty();
}

//-----------------------------------------------------------------------------------
ParameterValue::ParameterValue(const float v)
{
	setFloat(v);
}

//-----------------------------------------------------------------------------------
ParameterValue::ParameterValue(const int v)
{
	setInt(v);
}

//-----------------------------------------------------------------------------------
ParameterValue::ParameterValue(const bool v)
{
	setBool(v);
}

//-----------------------------------------------------------------------------------
ParameterValue::ParameterValue(const void* v)
{
	setPointer(v);
}

//-----------------------------------------------------------------------------------
ParameterValue::ParameterValue(const char* v)
{
	setString(v);
}

//-----------------------------------------------------------------------------------
ParameterValue::ParameterValue(const vec3f& v)
{
	setVector3(v);
}

//-----------------------------------------------------------------------------------
ParameterValue::ParameterValue(const vec4f& v)
{
	setVector4(v);
}

//-----------------------------------------------------------------------------------
ParameterValue::ParameterValue(const std::string& v)
{
	setString(v);
}

//-----------------------------------------------------------------------------------
ParameterValue::ParameterValue(const mat44f& v)
{
	setMatrix44(v);
}

//-----------------------------------------------------------------------------------
std::string ParameterValue::asString() const
{
	if (_any.empty())
		return "";
	return _any.operator()<std::string>();
}
//-----------------------------------------------------------------------------------
float ParameterValue::asFloat() const
{
	if (_any.empty())
		return 0.f;
	return _any.operator()<float>();
}

//-----------------------------------------------------------------------------------
bool ParameterValue::asBool() const
{
	if (_any.empty())
		return false;
	return _any.operator()<bool>();
}

//-----------------------------------------------------------------------------------
int ParameterValue::asInt() const
{
	if (_any.empty())
		return 0;
	return _any.operator()<int>();
}

//-----------------------------------------------------------------------------------
void* ParameterValue::asPointer() const
{
	if (_any.empty())
		return 0;
	return (void*)_any.operator()<unsigned char*>();
}

//-----------------------------------------------------------------------------------
vec4f ParameterValue::asVector4() const
{
	if (_any.empty())
		return make_vec4f(0,0,0,0);
	return _any.operator()<vec4f>();
}

//-----------------------------------------------------------------------------------
vec3f ParameterValue::asVector3() const
{
	if (_any.empty())
		return make_vec3f(0,0,0);
	return _any.operator()<vec3f>();
}

//-----------------------------------------------------------------------------------
mat44f ParameterValue::asMatrix44() const
{
	if (_any.empty())
		return mat44f::identity();
	return _any.operator()<mat44f>();
}

//-----------------------------------------------------------------------------------
mat33f ParameterValue::asMatrix33() const
{
	if (_any.empty())
		return mat33f::identity();
	return _any.operator()<mat33f>();
}

//-----------------------------------------------------------------------------------
void ParameterValue::setPointer(const void* v)
{
	_any = (unsigned char*)v;
}

//-----------------------------------------------------------------------------------
void ParameterValue::setString(const std::string& v)
{
	_any = v;
}

//-----------------------------------------------------------------------------------
void ParameterValue::setMatrix33(const mat33f& v)
{
	_any = v;
}

//-----------------------------------------------------------------------------------
void ParameterValue::setMatrix44(const mat44f& v)
{
	_any = v;
}

//-----------------------------------------------------------------------------------
void ParameterValue::setVector4(const vec4f& v)
{
	_any = v;
}

//-----------------------------------------------------------------------------------
void ParameterValue::setVector3(const vec3f& v)
{
	_any = v;
}

//-----------------------------------------------------------------------------------
void ParameterValue::setFloat(const float v)
{
	_any = v;
}

//-----------------------------------------------------------------------------------
void ParameterValue::setInt(const int v)
{
	_any = v;
}

//-----------------------------------------------------------------------------------
void ParameterValue::setBool(const bool v)
{
	_any = v;
}


//-----------------------------------------------------------------------------------
const Parameters& ParameterSpec::getMeta() const
{
	return _meta;
}

//-----------------------------------------------------------------------------------
int ParameterSpec::getTypeSize() const
{
	switch (_type)
	{
	case VEC3:
		return 3*sizeof(float);
	case VEC4:
		return 4*sizeof(float);
	case MATRIX:
		return 4*4*sizeof(float);
	case PTR:
		return sizeof(void*);
	case BOOL:
		return sizeof(bool);
	case INT:
		return sizeof(int);
	case FLOAT:
		return sizeof(float);
	case STRING:
		return sizeof(void*);
	default:
		return 0;
	}
}

//-----------------------------------------------------------------------------------
std::string ParameterSpec::getTypeName() const
{
	switch (_type)
	{
	case PTR:
		return "pointer";
	case BOOL:
		return "bool";
	case INT:
		return "int";
	case FLOAT:
		return "float";
	case VEC3:
		return "vec3";
	case VEC4:
		return "vec4";
	case STRING:
		return "string";
	case MATRIX:
		return "matrix";
	default:
		return "invalid";
	}
}

//-----------------------------------------------------------------------------------
std::string ParameterSpec::getName() const
{
	return _name;
}

//-----------------------------------------------------------------------------------
int ParameterSpec::getType() const
{
	return (int)_type;
}

//-----------------------------------------------------------------------------------
bool ParameterSpec::valid() const
{
	return (_type != INVALID);
}

Parameters::Parameters()
{
    specs = 0;
}

//-----------------------------------------------------------------------------------
void Parameters::setSpecs(const std::string& prefix, ParameterSpecs* s)
{
	specPrefix = prefix;
	specs = s;
}

//-----------------------------------------------------------------------------------
void Parameters::append(const Parameters& parameters, const std::string& prefix)
{
    Parameters::const_iterator it = parameters.begin();
    for (; it != parameters.end(); ++it)
    {
        Parameters::iterator thisIt = find(prefix + it->first);
        if (thisIt == end())
            insert(std::make_pair(prefix + it->first, it->second));
        else
            thisIt->second = it->second;
    }
}

//-----------------------------------------------------------------------------------
ParameterValuePtr Parameters::inputValue(const std::string& name, const ParameterValue& def)
{
    const_iterator it = find(name);

    if (it == end())
	{
		if (def.valid())
			return new ParameterValue(def);
		assert(0 && "Can't find ParameterValue in Parameters");
		return ParameterValuePtr();
	}

	return it->second;
}

//-----------------------------------------------------------------------------------
ParameterValuePtr Parameters::inputValue(const ParameterSpec* spec, const ParameterValue& def)
{
	if (!specs || !spec)
	{
		if (def.valid())
			return new ParameterValue(def);
		assert(0 && "Can't find ParameterValue in Parameters");
		return ParameterValuePtr();
	}

	std::string name = spec->_name;

    const_iterator it = find(specPrefix+name);

    if (it == end())
	{
        ParameterSpecs::const_iterator specIt = specs->find(name);
		// if we got the key...
		assert(specIt != specs->end());
		// use the default if specified...
		if (def.valid())
			return new ParameterValue(def);

		// otherwise return the spec default...
		return new ParameterValue(specIt->second->_defaultValue);
	}
	return it->second;
}

//-----------------------------------------------------------------------------------
Parameters& Parameters::set(std::string name, const ParameterValue& p)
{
	insert(std::make_pair(name, new ParameterValue(p)));
	return *this;
}

//-----------------------------------------------------------------------------------
bool Parameters::contains(std::string name) const
{
    const_iterator it = find(name);
    return (it != end());
}

//-----------------------------------------------------------------------------------
ParameterValue& Parameters::operator[] (std::string name)
{
    const_iterator it = find(name);
    ParameterValuePtr v;
	if (it == end())
	{
		v = new ParameterValue();
        insert(std::make_pair(name, v));
	}
    else
        v = it->second;
	return *v;
}

//-----------------------------------------------------------------------------------
const ParameterValue& Parameters::operator[] (std::string name) const
{
    const_iterator it = find(name);
	if (it == end())
		abort();
	return *it->second;
}

//-----------------------------------------------------------------------------------
std::string Parameters::asString(std::string name, std::string def) const
{
    const_iterator it = find(name);
	if (it == end())
		return def;
	return it->second->asString();
}
//-----------------------------------------------------------------------------------
float Parameters::asFloat(std::string name, float def) const
{
	const_iterator it = find(name);
	if (it == end())
		return def;
	return it->second->asFloat();
}

//-----------------------------------------------------------------------------------
bool Parameters::asBool(std::string name, bool def) const
{
	const_iterator it = find(name);
	if (it == end())
		return def;
	return it->second->asBool();
}

//-----------------------------------------------------------------------------------
int Parameters::asInt(std::string name, int def) const
{
	const_iterator it = find(name);
	if (it == end())
		return def;
	return it->second->asInt();
}

//-----------------------------------------------------------------------------------
void* Parameters::asPointer(std::string name, void* def) const
{
	const_iterator it = find(name);
	if (it == end())
		return def;
	return it->second->asPointer();
}

//-----------------------------------------------------------------------------------
vec4f Parameters::asVector4(std::string name, vec4f def) const
{
	const_iterator it = find(name);
	if (it == end())
		return def;
	return it->second->asVector4();
}

//-----------------------------------------------------------------------------------
vec3f Parameters::asVector3(std::string name, vec3f def) const
{
	const_iterator it = find(name);
	if (it == end())
		return def;
	return it->second->asVector3();
}

//-----------------------------------------------------------------------------------
mat44f Parameters::asMatrix44(std::string name, const mat44f& def) const
{
	const_iterator it = find(name);
	if (it == end())
		return def;
	return it->second->asMatrix44();
}

//------------------------------------------------------------------------------------------
ParameterSpecPtr HasParameterSpecs::defineParameter(const std::string& name, int type, const ParameterValue& def, const Parameters& meta)
{
	ParameterSpecPtr p = new ParameterSpec();
	p->_type = ParameterSpec::TypeType(type);
	p->_name = name;
	p->_meta = meta;
    p->_defaultValue = def;

    // this overwrites any existing definition.
	_parameterDefinitions[name] = p;

	return p;
}

//------------------------------------------------------------------------------------------
ParameterSpecPtr HasParameterSpecs::getParameter(const std::string& name)
{
    ParameterSpecs::const_iterator it = _parameterDefinitions.find(name);
    if (it == _parameterDefinitions.end())
        return ParameterSpecPtr();
    return it->second;
}

//-----------------------------------------------------------------------------------
}
}
