/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#ifndef NVPARTICLES_PARAMETERS_H_INCLUDED
#define NVPARTICLES_PARAMETERS_H_INCLUDED

//#pragma warning(push)
//#pragma warning(disable:4251)

#include "NvParticlesTypes.h"
#include "math_utils.h"
#include <algorithm>
#include <typeinfo>

namespace Easy
{

//----------------------------------------------------------------------------------------------
/// Variant type that can hold Any other type.
///
class _NvParticlesExport Any
{
public:

    Any()
        : mContent(0)
    {
    }

    template<typename ValueType>
    explicit Any(const ValueType & value)
        : mContent(new Holder<ValueType>(value))
    {
    }

    Any(const Any & other)
        : mContent(other.mContent ? other.mContent->clone() : 0)
    {
    }

    virtual ~Any()
    {
        destroy();
    }

public:

    Any& swap(Any & rhs)
    {
        std::swap(mContent, rhs.mContent);
        return *this;
    }

    template<typename ValueType>
    Any& operator=(const ValueType & rhs)
    {
        Any(rhs).swap(*this);
        return *this;
    }

    Any & operator=(const Any & rhs)
    {
        Any(rhs).swap(*this);
        return *this;
    }

public: // queries

    bool empty() const
    {
        return !mContent;
    }

    const std::type_info& type() const
    {
        return mContent ? mContent->type() : typeid(void);
    }

    inline friend bool operator==(const Any& a, const Any& b)
    {
        if(a.mContent == b.mContent)
            return true;
        if(!a.mContent || !b.mContent)
            return false;
        return (a.mContent->compare(*b.mContent) != 0);
    }

    inline friend bool operator!=(const Any& a, const Any& b)
    {
        return !(a == b);
    }

    void destroy()
    {
        delete(mContent);
        mContent = NULL;
    }

protected: // types

    class Placeholder
    {
    public: // constructors

        virtual ~Placeholder()
        {
        }

    public: // queries

        virtual const std::type_info& type() const = 0;

        virtual Placeholder *clone() const = 0;

        virtual int compare(Placeholder& a) const = 0;

        inline friend int operator==(const Placeholder& a, Placeholder& b)
        {
            return a.compare(b);
        }
    };

    template<typename ValueType>
    class Holder : public Placeholder
    {
    public: // constructors

        Holder(const ValueType & value)
            : held(value)
        {
        }

    public: // queries

        virtual const std::type_info & type() const
        {
            return typeid(ValueType);
        }

        virtual Placeholder *clone() const
        {
            return new(Holder)(held);
        }

        virtual int compare(Placeholder& a) const
        {
            return held == static_cast<Any::Holder<ValueType> *>(&a)->held;
        }

    public:

        ValueType held;
    };

protected:
    Placeholder* mContent;

    template<typename ValueType>
    friend ValueType * any_cast(Any *);

public:

    template<typename ValueType>
    ValueType operator()() const
    {
        if (!mContent)
        {
            std::cerr << "Error casting from uninitialised Any" << std::endl;
            abort();
        }
        else if(type() != typeid(ValueType))
        {
            std::cerr << "Error casting from \"" << type().name() << "\" " << "to \"" << typeid(ValueType).name() << "\"" << std::endl;
        }

		return static_cast<Any::Holder<ValueType> *>(mContent)->held;
    }
};

//----------------------------------------------------------------------------------------------
template<typename ValueType>
ValueType* any_cast(Any * operand)
{
    return operand && operand->type() == typeid(ValueType)
           ? &static_cast<Any::Holder<ValueType> *>(operand->mContent)->held
           : 0;
}

//----------------------------------------------------------------------------------------------
template<typename ValueType>
const ValueType * any_cast(const Any * operand)
{
    return any_cast<ValueType>(const_cast<Any *>(operand));
}

//----------------------------------------------------------------------------------------------
template<typename ValueType>
ValueType any_cast(const Any & operand)
{
    const ValueType * result = any_cast<ValueType>(&operand);
    if(!result)
    {
        std::cerr << "Error casting from '" << operand.type().name() << "' " << "to '" << typeid(ValueType).name() << "'";
		abort();
    }
    return *result;
}


namespace NvParticles
{

//-----------------------------------------------------------------------------------
class _NvParticlesExport ParameterValue
{
public:
	ParameterValue();
	ParameterValue(const ParameterValue& b);
	ParameterValue& operator=(const ParameterValue& b);
	ParameterValue(const float v);
	ParameterValue(const int v);
	ParameterValue(const bool v);
	ParameterValue(const void* v);
	ParameterValue(const char* v);
	ParameterValue(const vec3f& v);
	ParameterValue(const vec4f& v);
	ParameterValue(const std::string& v);
	ParameterValue(const mat44f& v);

	bool valid() const;

	mat33f asMatrix33() const;
	mat44f asMatrix44() const;
	vec3f asVector3() const;
	vec4f asVector4() const;
	void* asPointer() const;
	int asInt() const;
	bool asBool() const;
	float asFloat() const;
    std::string asString() const;

    void setPointer(const void* v);
	void setString(const std::string& s);
	void setMatrix33(const mat33f& v);
	void setMatrix44(const mat44f& v);
	void setVector4(const vec4f& v);
	void setVector3(const vec3f& v);
	void setFloat(const float v);
	void setInt(const int v);
	void setBool(const bool v);

	friend int operator==(const ParameterValue& a, const ParameterValue& b)
	{
		return (a._any == b._any);
	}

	friend int operator!=(const ParameterValue& a, const ParameterValue& b)
	{
		return !(a==b);
	}

protected:
	Any _any;
};

class ParameterSpec;

typedef SharedPtr<ParameterValue> ParameterValuePtr;

//-----------------------------------------------------------------------------------
typedef SharedPtr<ParameterSpec> ParameterSpecPtr;
typedef std::map<std::string, ParameterSpecPtr > ParameterSpecs;

//-----------------------------------------------------------------------------------
/// class for the parameter block. (i.e. the parameter values)
///
class _NvParticlesExport Parameters : public std::map<std::string, ParameterValuePtr>
{
    friend class ParameterSpec;

	ParameterSpecs* specs;
	std::string specPrefix;

public:
    Parameters();

	void setSpecs(const std::string& specPrefix, ParameterSpecs* specs);

    /// Add more parameters with optional prefix.
    void append(const Parameters& parameters, const std::string& prefix="");

	ParameterValuePtr inputValue(const std::string& name, const ParameterValue& def=ParameterValue());
	ParameterValuePtr inputValue (const ParameterSpec* key, const ParameterValue& def=ParameterValue());
	ParameterValuePtr outputValue (const ParameterSpec* key);

    /// Get the parameter with this name, and create it if it doesn't exist.
	ParameterValue& operator[] (std::string name);
	const ParameterValue& operator[] (std::string name) const;

    /// Set a parameter.
	Parameters& set(std::string name, const ParameterValue& p);

    bool contains(std::string name) const;

    std::string asString(std::string name, std::string def="") const;
    float asFloat(std::string name, float def=0.f) const;
    int asInt(std::string name, int def=0) const;
    bool asBool(std::string name, bool def=false) const;
    void* asPointer(std::string name, void* def=0) const;
    vec3f asVector3(std::string name, vec3f def=make_vec3f(0)) const;
    vec4f asVector4(std::string name, vec4f def=make_vec4f(0)) const;
	mat44f asMatrix44(std::string name, const mat44f& def=mat44f::identity()) const;
};

//-----------------------------------------------------------------------------------
class _NvParticlesExport ParameterSpec
{
public:
	enum TypeType {INVALID=0,INT,FLOAT,BOOL,PTR,VEC3,VEC4,STRING,MATRIX,TEXTURE};

public:

	ParameterSpec()
	{
		_type = INVALID;
	}

	ParameterSpec& operator=(const ParameterSpec& p)
	{
		/// CAVEAT: don't copy the name as this is passed in from the Parameters class!
		//_name = p._name;
		_meta = p._meta;
		_type = p._type;
		_defaultValue = p._defaultValue;
		return (*this);
	}

	friend inline int operator==(const ParameterSpec& a, const ParameterSpec& b)
    {
        return (!(a != b));
    }

	friend inline bool operator!=(const ParameterSpec& a, const ParameterSpec& b)
	{
        if (a._name != b._name)
			return true;
        /*
		if (a._type != b._type)
			return true;
		if (a._defaultValue != b._defaultValue)
			return true;
		if (a._meta != b._meta)
			return true;*/
        return false;
	}

	ParameterSpec(TypeType t, const Parameters& m)
	{
		_type = t;
		_meta = m;
	}

	bool valid() const;

    std::string getName() const;
	const Parameters& getMeta() const;

    int getType() const;
	int getTypeSize() const;
	std::string getTypeName() const;

protected:
	std::string _name;
	Parameters _meta;
	TypeType _type;
public:
    /// TODO: make this private.
	ParameterValue _defaultValue;

    friend class Parameters;
    friend class HasParameterSpecs;
};

//----------------------------------------------------------------------------------------------
class _NvParticlesExport HasParameterSpecs
{
protected:

	ParameterSpecs _parameterDefinitions;

	ParameterSpecPtr defineParameter(const std::string& name, int type, const ParameterValue& def, const Parameters& meta=Parameters());

	ParameterSpecPtr getParameter(const std::string& name);

public:

	ParameterSpecs& parameterDefinitions()
	{
		return _parameterDefinitions;
	}
};

//----------------------------------------------------------------------------------------------
}
}

//#pragma warning(pop)

#endif // NVPARTICLES_PARAMETERS_H_INCLUDED
