/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#include "NvParticlesParticleBuffer.h"

namespace Easy
{
namespace NvParticles
{

//------------------------------------------------------------------------------------------
ParticleBufferSpec::ParticleBufferSpec()
	:
    type(NONE),
    flags(0),
    elementBytes(0)
{
}

//------------------------------------------------------------------------------------------
ParticleBufferSpec::ParticleBufferSpec(const std::string& _name, const ParticleBufferSpec::Type _type, const int _flags, std::string _renderSemantic)
	:
	name(_name),
    type(_type),
    flags(_flags),
    elementBytes(0),
    renderSemantic(_renderSemantic)
{
    if(type == FLOAT)
        elementBytes = sizeof(float);
    else if(type == FLOAT4)
        elementBytes = sizeof(float)*4;
    else if(type == UINT)
        elementBytes = sizeof(unsigned int);
    else if(type == MAT33F)
        elementBytes = sizeof(mat33f);
    else
    {
        printError(Stringf("ParticleBufferSpec: type not yet implemented: %d", type));
    }

    if (_renderSemantic.length() == 0)
        renderSemantic = name;
}

//------------------------------------------------------------------------------------------
ParticleBuffer::ParticleBuffer(const ParticleBufferSpec& _spec)
    :
    spec(_spec),
    _deviceMapper(Cu::Buffer::CUDA),
	_hostMapper(Cu::Buffer::HOST),
    _lockLevel(0),
    _dirty(false),
	_current(0),
	_buffers(0),
	_nBuffers(0),
    _currentUpdateState(0)
{
	_nBuffers = 1;
	if (spec.flags & ParticleBufferSpec::DOUBLEBUFFER)
		_nBuffers = 2;
	_buffers = new Cu::Buffer[_nBuffers];
}

ParticleBuffer::~ParticleBuffer()
{
	delete[] _buffers;
}

//------------------------------------------------------------------------------------------
size_t ParticleBuffer::copy(const ParticleBuffer& src, size_t doff, size_t soff, size_t count, const Cu::Buffer::CopyOptions& options)
{
    _dirty = true;
    return buffer()->Copy(*src.buffer(), doff*spec.elementBytes, soff*spec.elementBytes, count*spec.elementBytes, options);
}

//------------------------------------------------------------------------------------------
int operator==(const ParticleBufferSpec& a, const ParticleBufferSpec& b)
{
    return ((a.name == b.name));// && (a.type == b.type) && (a.flags == b.flags));// && (a.renderSemantic == b.renderSemantic));
}

//------------------------------------------------------------------------------------------
int operator!=(const ParticleBufferSpec& a, const ParticleBufferSpec& b)
{
    return ((a.name != b.name));// || (a.type != b.type) || (a.flags != b.flags));
}

//------------------------------------------------------------------------------------------
unsigned long long ParticleBuffer::getUpdateId()
{
    return _currentUpdateState;
}

void ParticleBuffer::setUpdateId(unsigned long long v)
{
    _currentUpdateState = v;
}

//------------------------------------------------------------------------------------------
void ParticleBuffer::lock(long stream)
{
    if (_lockLevel>0)
        printError("Already locked: " + spec.name);

    ++_lockLevel;

    if (_lockLevel == 1)
    {
        if (_deviceMapper.Valid())
        {
            printError("Mapper already Valid: " + spec.name);
            return;
        }

        Cu::Buffer::MapOptions options;
        options.SetStream(stream);
        _deviceMapper.Map(_buffers[_current], options);
    }
}

//------------------------------------------------------------------------------------------
void ParticleBuffer::unlock()
{
    if (_lockLevel==0)
        printError("Not yet locked: " + spec.name);

    --_lockLevel;

    if (_lockLevel==0)
    {
        if(!_deviceMapper.Valid())
        {
            printError("Mapper is invalid: " + spec.name);
            return;
        }
        _deviceMapper.Unmap();
    }
}

//------------------------------------------------------------------------------------------
void* ParticleBuffer::devicePointer() const
{
    if (!_deviceMapper.Valid())
    {
        if(_buffers[_current].Type() == Cu::Buffer::CUDA)
            return _buffers[_current].Data();

        printError("Must lock device buffer: " + spec.name);
        return 0;
    }
    return (void*)_deviceMapper.Pointer();
}

//------------------------------------------------------------------------------------------
void* ParticleBuffer::hostPointer() const
{
    if (!_hostMapper.Valid())
    {
        if(_buffers[_current].Type() == Cu::Buffer::HOST)
            return _buffers[_current].Data();

        printError("Must lock host buffer: " + spec.name);
        return NULL;
    }
    return (void*)_hostMapper.Pointer();
}

//------------------------------------------------------------------------------------------
const Cu::Buffer& ParticleBuffer::operator[] (int i) const
{
    return _buffers[(_current+i)%_nBuffers];
}

//------------------------------------------------------------------------------------------
Cu::Buffer& ParticleBuffer::operator[] (int i)
{
    return _buffers[(_current+i)%_nBuffers];
}

//------------------------------------------------------------------------------------------
int ParticleBuffer::page() const
{
    return _current;
}

//------------------------------------------------------------------------------------------
void ParticleBuffer::flip()
{
	if(--_current < 0)
		_current = _nBuffers-1;
}

//------------------------------------------------------------------------------------------
Cu::Buffer* ParticleBuffer::buffer()
{
    if (_deviceMapper.Valid())
        return &_deviceMapper.internal;
    return &_buffers[_current];
}

//------------------------------------------------------------------------------------------
const Cu::Buffer* ParticleBuffer::buffer() const
{
    if (_deviceMapper.Valid())
        return &_deviceMapper.internal;
    return &_buffers[_current];
}

//------------------------------------------------------------------------------------------
bool ParticleBuffer::allocate(int n)
{
    if (n == 0)
        return false;
    _dirty = true;
    Cu::Buffer::MemType t = Cu::Buffer::HOST;
	bool rc = true;
	for(int i=0;rc && i<_nBuffers;++i)
		rc = _buffers[i].Allocate(t, spec.elementBytes*n, 0, spec.name.c_str());
    return rc;
}

//------------------------------------------------------------------------------------------
void ParticleBuffer::clear(int v)
{
	for(int i=0;i<_nBuffers;++i)
		_buffers[i].Clear(v);
}

//------------------------------------------------------------------------------------------
void ParticleBuffer::dump(const char* msg, size_t count, size_t step) const
{
    std::string name = spec.name;
    if (msg)
        name = std::string(msg) + " " + name;

    if (!spec.renderSemantic.empty())
    {
        name += " (semantic = " + spec.renderSemantic + ")";
    }

    if(spec.type == ParticleBufferSpec::FLOAT4)
        _buffers[_current].DumpAs<vec4f>(name.c_str(), count, step);
    else if(spec.type == ParticleBufferSpec::FLOAT)
        _buffers[_current].DumpAs<float>(name.c_str(), count, step);
    else if(spec.type == ParticleBufferSpec::UINT)
        _buffers[_current].DumpAs<unsigned int>(name.c_str(), count, step);
    else if(spec.type == ParticleBufferSpec::MAT33F)
        _buffers[_current].DumpAs<mat33f>(name.c_str(), count, step);
}

//------------------------------------------------------------------------------------------
void HasParticleBuffers::setBuffer(const std::string& name, const ParticleBufferPtr& buffer)
{
    //ScopedMutex _sm(_bufferLock);
    _buffers.insert(make_pair(name, buffer));
}

//------------------------------------------------------------------------------------------
ParticleBufferPtr HasParticleBuffers::getBuffer(const std::string& name)
{
    //ScopedMutex _sm(_bufferLock);
    ParticleBuffers::const_iterator it = _buffers.find(name);

    if (it != _buffers.end())
        return it->second;

    return ParticleBufferPtr();
}

//------------------------------------------------------------------------------------------
const ParticleBufferPtr& HasParticleBuffers::getBuffer(const std::string& name) const
{
    ParticleBuffers::const_iterator it = _buffers.find(name);

    if (it != _buffers.end())
        return it->second;

    printError("Failed to find buffer: " + name);
    abort();

    return it->second;
}

//------------------------------------------------------------------------------------------
const ParticleBuffers& HasParticleBuffers::getBuffers() const
{
    return _buffers;
}

//------------------------------------------------------------------------------------------
void HasParticleBuffers::dump(size_t count, size_t step)
{
    //ScopedMutex _sm(_bufferLock);

    // allocate a sorted variant of all the persistent buffers...
    for (ParticleBuffers::const_iterator it=_buffers.begin(); it != _buffers.end(); ++it)
    {
        printf("---------------------------------------\n");
        it->second->dump("", count, step);
    }
}

//------------------------------------------------------------------------------------------
}
}
