/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#ifndef NVPARTICLESPARTICLEBUFFER_H_INCLUDED
#define NVPARTICLESPARTICLEBUFFER_H_INCLUDED

#include "NvParticlesTypes.h"

namespace Easy
{
namespace NvParticles
{

//-----------------------------------------------------------------------------------
class _NvParticlesExport ParticleBufferSpec
{
public:
    enum Type {NONE=0, FLOAT, FLOAT4, UINT, MAT33F};
    enum Flags {DYNAMIC=0, STATIC=1, RENDERABLE=16, PERSISTENT=128, DOUBLEBUFFER=256};

    std::string name;
    Type type;
    int flags;
    size_t elementBytes;
    std::string renderSemantic;

public:
    ParticleBufferSpec();
    ParticleBufferSpec(const std::string& name, const Type type, const int flags=0, std::string _renderSemantic="");

	friend int operator==(const ParticleBufferSpec& a, const ParticleBufferSpec& b);
	friend int operator!=(const ParticleBufferSpec& a, const ParticleBufferSpec& b);
};

typedef std::map<std::string, ParticleBufferSpec> ParticleBufferSpecs;

//-----------------------------------------------------------------------------------
class _NvParticlesExport ParticleBuffer
{
public:
    ParticleBuffer(const ParticleBufferSpec& spec);
    ~ParticleBuffer();

    void dump(const char* msg="", size_t count=size_t(-1), size_t step=1) const;
    bool allocate(int n);
    void lock(long stream=0);
    void unlock();
    void* devicePointer() const;
	void* hostPointer() const;
	void* pointer() const;
    Cu::Buffer* buffer();
    const Cu::Buffer* buffer() const;

	void clear(int v);

    size_t copy(const ParticleBuffer& src, size_t doff=0, size_t soff=0, size_t count=size_t(-1), const Cu::Buffer::CopyOptions& options=Cu::Buffer::DefaultCopyOptions);

	// multi-page methods:
    const Cu::Buffer& operator[] (int i) const;
    Cu::Buffer& operator[] (int i);
    void flip();
    int page() const;

    unsigned long long getUpdateId();
    void setUpdateId(unsigned long long v);

    ParticleBufferSpec spec;

private:

    int _lockLevel;
    Cu::BufferMapper<unsigned char> _deviceMapper;
	Cu::BufferMapper<unsigned char> _hostMapper;

    bool _dirty;

    Cu::Buffer* _buffers;
	int _nBuffers;
	int _current;
    unsigned long long _currentUpdateState;

    friend class ParticleModifier;
};

typedef SharedPtr<ParticleBuffer> ParticleBufferPtr;
typedef std::map<std::string, ParticleBufferPtr> ParticleBuffers;

//-----------------------------------------------------------------------------------
class _NvParticlesExport HasParticleBuffers
{
public:
    /// Get the per-particle buffer.
    /// @param name The name of the buffer
    ///
    ParticleBufferPtr getBuffer(const std::string& name);
    const ParticleBufferPtr& getBuffer(const std::string& name) const;

	/// @brief		Dump the buffers.
	///
    void dump(size_t count=size_t(-1), size_t step=1);

    void setBuffer(const std::string& name, const ParticleBufferPtr& buffer);

    const ParticleBuffers& getBuffers() const;

protected:

    ParticleBuffers _buffers;
    Mutex _bufferLock;
};

//-----------------------------------------------------------------------------------
}
}

#endif // NVPARTICLESPARTICLEBUFFER_H_INCLUDED
