/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#include "NvParticlesParticleContainer.h"
#include "NvParticlesConfig.h"
#include "NvParticlesManager.h"
#include "cuda_utils.h"

namespace Easy
{
namespace NvParticles
{

int ParticleContainer::debugLevel = 0;

//------------------------------------------------------------------------------------------
ParticleContainer::ParticleContainer()
{
	maxParticles = 0;

    _init();

    _currentUpdateState = requestedUpdateState = 0;

    clear();
}

//-----------------------------------------------------------------------------------
void ParticleContainer::_init()
{
    enableSimulation = true;
    initialized = false;
    numParticles = 0;
    enableSimulation = true;
    initialized = false;
    dirty = false;

    // we need to ensure these are reset properly, so we always create valid bounds.
    // i.e. high >= low (for all valid bounds)
	boundsLow = make_vec3f(+FLT_MAX);
	boundsHigh = make_vec3f(-FLT_MAX);
}

//-----------------------------------------------------------------------------------
unsigned long long ParticleContainer::getUpdateId()
{
    return _currentUpdateState;
}

//-----------------------------------------------------------------------------------
void ParticleContainer::setMaxParticles(int n)
{
    if (maxParticles != n)
	{
        // wipe all the existing buffers.
        clear();
        maxParticles = n;
	}
}

//-----------------------------------------------------------------------------------
int ParticleContainer::getParticleCount()
{
    return numParticles;
}

//------------------------------------------------------------------------------------------
void ParticleContainer::clear()
{
	_buffers.clear();
    bufferSpecs.clear();

    numParticles = 0;
    dirty = false;
    _firstTime = true;

    // we need to ensure these are reset properly, so we always create valid bounds.
    // i.e. high >= low (for all valid bounds)
	boundsLow = make_vec3f(+FLT_MAX);
	boundsHigh = make_vec3f(-FLT_MAX);
}

//------------------------------------------------------------------------------------------
ParticleBufferPtr ParticleContainer::_createBuffer(const ParticleBufferSpec& spec)
{
    printInfo(Stringf("particle-container - allocating: %s", spec.name.c_str()));

    ParticleBufferPtr buf = new ParticleBuffer(spec);

    bool rc = buf->allocate(maxParticles);
    if (!rc)
	{
		printError(Stringf("particle-container - unable to allocate buffer: %s", spec.name.c_str()));
        buf.setNull();
	}
	else
	{
        // zero the buffer.
		buf->clear(0);
	}

    return buf;
}

//------------------------------------------------------------------------------------------
bool ParticleContainer::addBuffer(const ParticleBufferSpec& spec)
{
    if (!maxParticles)
        return false;

    bool rc = false;

    _bufferLock.claim();

	ParticleBufferPtr buf;

    buf = getBuffer(spec.name);

	if (!buf || buf->spec != spec)
	{
		buf = _createBuffer(spec);

		if (buf.valid())
        {
            setBuffer(spec.name, buf);
            rc = true;
        }
	}

    _bufferLock.release();

    return rc;
}

//-----------------------------------------------------------------------------------
void ParticleContainer::emitContainer(ParticleContainer* masterContainer, float birthTime, float birthTimeJitter)
{
    if (!maxParticles)
        return;

    assert(masterContainer);
    if(masterContainer->numParticles == 0)
        return;

    float* p_birthTime = 0;
    if (getBuffer("birthTime"))
	{
    	p_birthTime = (float*)getBuffer("birthTime")->buffer()->Data();

        for(int i=0; i<masterContainer->numParticles; ++i)
        {
            unsigned int seed = numParticles+i;
            p_birthTime[numParticles+i] = birthTime + birthTimeJitter*(random(seed)*2-1);
        }
    }

    if (masterContainer->getBuffer("position"))
    {
        float4* p_position = 0;
        if(getBuffer("position"))
		    p_position = (float4*)getBuffer("position")->buffer()->Data();

        float4* p_positionSrc = (float4*)masterContainer->getBuffer("position")->buffer()->Data();

        for(int i=0; i<masterContainer->numParticles; ++i)
        {
            vec3f pos = make_vec3f(p_positionSrc[i].x, p_positionSrc[i].y, p_positionSrc[i].z);
            p_position[numParticles+i] = ::make_float4(pos.x, pos.y, pos.z, 3);
        }
    }

    // copy all the other buffers
    for (ParticleBuffers::const_iterator bit=masterContainer->getBuffers().begin(); bit != getBuffers().end(); ++bit)
    {
        if(bit->second->spec.name == "position" || bit->second->spec.name == "birthTime" || bit->second->spec.name == "id")
            continue;

        ParticleBuffer* destBuf = getBuffer(bit->second->spec.name.c_str());
        destBuf->copy(*bit->second.pointer(), numParticles, 0, masterContainer->numParticles);
    }

	if(debugLevel)
        printInfo(Stringf("particle-container emission - offset(%d) count(%d)", numParticles, masterContainer->numParticles));

    numParticles = numParticles + masterContainer->numParticles;

    dirty = true;
}

//------------------------------------------------------------------------------------------
/// Emit data into the pending buffers.
/// Positions and velocities are provided in worldScale.
/// However velocities need to be converted to the simulation scale.
///
void ParticleContainer::emitBuffers(int n, float birthTime, double* posd, double* vel, double* timestep, double* rgb, int rgbN, double* opacity)
{
    if (!maxParticles)
        return;

    // are we already full?
    if(numParticles >= maxParticles)
        return;

    // trim the count.
    if(numParticles + n >= maxParticles)
        n = maxParticles - numParticles;

	float4* p_position = 0;
	float4* p_velocity = 0;
	float4* p_color = 0;
    float* p_birthTime = 0;

	if(getBuffer("position"))
		p_position = (float4*)getBuffer("position")->buffer()->Data();
	if(getBuffer("velocity"))
	    p_velocity = (float4*)getBuffer("velocity")->buffer()->Data();
	if(getBuffer("color"))
		p_color = (float4*)getBuffer("color")->buffer()->Data();
	if(getBuffer("birthTime"))
		p_birthTime = (float*)getBuffer("birthTime")->buffer()->Data();

	uint counter = numParticles;

	float positionScale = 1;
	float velocityScale = 1;

    for (uint i=0; i<n; i++)
    {

        vec3f pos = make_vec3f((float)posd[i*3+0], (float)posd[i*3+1], (float)posd[i*3+2]);

        double dt = timestep[i];
		if(p_position)
		{
			p_position[counter] = ::make_float4(pos.x, pos.y, pos.z, 0.f) * positionScale;
			p_position[counter].w = 3;
		}

		if(p_velocity)
			p_velocity[counter] = ::make_float4((float)vel[i*3+0], (float)vel[i*3+1], (float)vel[i*3+2], 0.f) * velocityScale;

		if(p_color)
		{
			int rgbi = i;
			if(rgbi >= rgbN)
				rgbi = 0;

			if(rgb)
			{
				p_color[counter] = ::make_float4(rgb[rgbi*3+0], rgb[rgbi*3+1], rgb[rgbi*3+2], 1);
			}
			else
				p_color[counter] = ::make_float4(1);

			if(opacity)
				p_color[counter].w = opacity[rgbi];
		}

        if(p_birthTime)
        {
            p_birthTime[counter] = birthTime + dt;
        }


        boundsLow = fminf(boundsLow, pos);
        boundsHigh = fmaxf(boundsHigh, pos);

        ++counter;
    }

	if (debugLevel > 0)
		printInfo(Stringf("particle-container emission - offset(%d) count(%d)", numParticles, counter-numParticles));

    numParticles = counter;

    dirty = true;
}

//-----------------------------------------------------------------------------------
void ParticleContainer::emitBox(const mat44f& xform, float spacing, float jitter,
    vec4f color,
    float birthTime, float birthTimeJitter)
{
    if (!maxParticles)
        return;

	float4* p_position = 0;
	float4* p_velocity = 0;
	float4* p_color = 0;
    float* p_birthTime = 0;

    if (!getBuffer("position"))
    {
        printError("particle container requires at least position for emission");
        return;
    }

	if (getBuffer("position"))
        p_position = (float4*)getBuffer("position")->hostPointer();
	if (getBuffer("velocity"))
	    p_velocity = (float4*)getBuffer("velocity")->hostPointer();
	if (getBuffer("color"))
		p_color = (float4*)getBuffer("color")->hostPointer();
	if (getBuffer("birthTime"))
		p_birthTime = (float*)getBuffer("birthTime")->hostPointer();

	//get the worldLengths... (multiply by 2 because we go from -1 to +1)
	float scalex = length(xform.X())*2;
	float scaley = length(xform.Z())*2;
	float scalez = length(xform.Y())*2;

    // jittering is a factor of spacing.
    jitter *= spacing;

	// method: 0 = regular-lattice, 1 = hexagonal-close-packing, 2 = cubic-close-packing...
	const int method = 1;

    // these counts are floored if we have extra space...
    int rx = scalex / spacing;
    int ry = scaley / spacing;
    int rz = scalez / spacing;

    // recalculate the spacing to make the gap between
    // particles the same.
    float dx = scalex / rx;
    float dy = scaley / ry;
    float dz = scalez / rz;

	if(method != 0)
	{
		// get the close-packing spacing for each dimension...
		dx = dx;
		dy = sqrtf((3.f*dx*dx)/4.f);
		dz = sqrtf((3.f*dx*dx)/4.f);

		// update the num particles in each dimension...
		rx = scalex / dx;
		ry = scaley / dy;
		rz = scalez / dz;

        // make sure that z-axis is a multiple of 3.
        ry += ry%3;
        // make sure that x-axis is a multiple of 2.
        rx += rx%2;

        // recalculate the spacing to make the gap between
        // particles the same.
        dx = scalex / rx;
        dy = scaley / ry;
        dz = scalez / rz;

        // put into unit space.
        dx = dx / scalex;
        dy = dy / scaley;
        dz = dz / scalez;
	}

	// shift-offsets...
	float offsetx;
	float offsety;
	float offsetz;

    float fx;
    float fy;
    float fz;

    uint i = numParticles;
	unsigned int seed = i;

    fz = dz / 2;
	for (uint z=0; z<rz; z++)
    {
		offsety = 0;

		if (method==1)
		{
			if(z%2 == 0)
				offsety = dy/3;
		}
		else if(method==2)
		{
			if(z%3 == 1)
				offsety = dy/3;
			else if(z%3 == 2)
				offsety = (dy/3)/2;
		}

		//fz = float(z)/(rz);

        fy = dy / 2 + offsety;

		for (uint y=0; y<ry; y++)
        {
			offsetx = 0;

			if (method==1)
			{
				if(y%2 == 0)
					offsetx = dx/2;
			}
			else if(method==2)
			{
				if(y%2 == 1)
					offsetx = dx/2;
				if(z%3 == 2)
					offsetx += dx/2;
			}

			//fy = float(y)/(ry);

            fx = offsetx + dx / 2;

            for (uint x=0; x<rx; x++)
            {
				// check we haven't reached our limit.
                if(i >= maxParticles)
				{
					x = rx;
					y = ry;
					z = rz;
					break;
				}

                offsetz = 0;

				//fx = float(x)/(rx);


                // jitter the position...
                float rfx = fx + (random(seed)*2.0f-1.0f)*(jitter/scalex);
                float rfy = fy + (random(seed)*2.0f-1.0f)*(jitter/scaley);
                float rfz = fz + (random(seed)*2.0f-1.0f)*(jitter/scalez);

				// create normalized position (-1 to +1).
				vec3f pos;
				pos.x = (rfx*2-1);
				pos.z = (rfy*2-1);
				pos.y = (rfz*2-1);

				// transform into world-space.
                pos = xform.multiplyPoint(pos);

				if(p_position)
				{
                    /// CAVEAT:
                    // 3 flag means it is brand-new and requires processing.
					p_position[i] = make_float4(pos.x, pos.y, pos.z, 3);
				}

				if(p_velocity)
				{
	                p_velocity[i] = make_float4(0.0f, 0.0f, 0.0f, 0.0f);
				}

				if(p_color)
				{
					p_color[i].x = color.x;
					p_color[i].y = color.y;
					p_color[i].z = color.z;
					p_color[i].w = color.w;
				}

                if(p_birthTime)
                {
                    p_birthTime[i] = birthTime + birthTimeJitter*(random(seed)*2-1);
                }

                boundsLow = fminf(boundsLow, pos);
                boundsHigh = fmaxf(boundsHigh, pos);

                i++;
                fx += dx;
			}
            fy += dy;
        }
        fz += dz;
    }

	if (debugLevel > 0)
        printInfo(Stringf("particle-container emission - offset(%d) count(%d)", numParticles, i-numParticles));

    numParticles = i;

    dirty = true;
}

//-----------------------------------------------------------------------------------
void ParticleContainer::setExportedBuffers(const StringArray& exportRequests)
{
    _exportRequests = exportRequests;
}

//-----------------------------------------------------------------------------------
bool ParticleContainer::isExportingBuffer(const std::string& name)
{
    if (!_exportRequests.contains(name))
        return false;
    return true;
}

//-----------------------------------------------------------------------------------
void ParticleContainer::updateExportBuffers(NvParticles::Parameters& outAttr)
{
    if (maxParticles == 0) // not yet initialized.
        return;

    for (int i=0; i<_exportRequests.size(); ++i)
    {
        std::string name = _exportRequests[i].c_str();
        ParticleBuffer* b = getBuffer(name);
        if (b == 0)
        {
            printError("Unable to export buffer: " + name);
            continue;
        }

        // prefix the attribute with "buffer." for finding it later.
        std::string name2 = Stringf("buffer.%s", _exportRequests[i].c_str());

        // assign the pointer.
        outAttr[name2].setPointer(b->hostPointer());
    }

    outAttr["numParticles"].setInt(numParticles);
}

//------------------------------------------------------------------------------------------
void ParticleContainer::_resetBuffers()
{
    for (ParticleBuffers::const_iterator it=_buffers.begin(); it != _buffers.end(); ++it)
    {
        if (it->first == "id")
        {
            // initialize IDs as sequential numbers...
            Cu::BufferMapper<uint> ptr(Cu::Buffer::HOST, *it->second->buffer());
            for (uint i=0; i<maxParticles; ++i)
                ptr[i] = i;
        }
        else
        {
            // otherwise just zero it.
            it->second->clear(0);
        }
    }
}

//-----------------------------------------------------------------------------------
void ParticleContainer::reset()
{
    _resetBuffers();
    boundsLow = make_vec3f(0);
	boundsHigh = make_vec3f(0);
    numParticles = 0;
    _firstTime = true;
}

//------------------------------------------------------------------------------------------
void ParticleContainer::getBounds(float* low, float* high)
{
    low[0] = boundsLow.x;
    low[1] = boundsLow.y;
    low[2] = boundsLow.z;
    high[0] = boundsHigh.x;
    high[1] = boundsHigh.y;
    high[2] = boundsHigh.z;
}

//------------------------------------------------------------------------------------------
}
}
