/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#ifndef NVPARTICLES_PARTICLE_CONTAINER_H_INCLUDED
#define NVPARTICLES_PARTICLE_CONTAINER_H_INCLUDED

#include "gl_utils.h"

#include "NvParticlesExports.h"
#include "NvParticlesTypes.h"
#include "NvParticlesParticleBuffer.h"
#include "NvParticlesParticleModifier.h"
#include "NvParticlesParticleSolverSpec.h"


namespace Easy
{
namespace Cu
{
class Stream;
}
//------------------------------------------------------------------------------------------
class CudaScheduler;

//------------------------------------------------------------------------------------------
namespace NvParticles
{
class Primitives;
class ParticleGrid;
class ParticleSolverImpl;
class ParticleForces;

//------------------------------------------------------------------------------------------
/// The data storage for the particles.
///
class _NvParticlesExport ParticleContainer : public HasParticleBuffers
{
public:

	static int debugLevel;

    ParticleContainer();

	/// Create a particle-buffer.
    ///
    bool addBuffer(const ParticleBufferSpec& spec);

	/// Clear all the data.
    ///
    void clear();

	/// Set the maximum number of particles.
	///
    void setMaxParticles(int n);

	/// Return the number of active particles.
	///
    int getParticleCount();

	/// Store the bounds in the triples.
	///
    void getBounds(float* low, float* high);

    /// Reset the container to an empty state.
    ///
    void reset();

    bool isExportingBuffer(const std::string& name);

    unsigned long long getUpdateId();

	/// @brief  Emit from an array of data.
	/// @detail The position, velocity, & color buffers must exist for us to emit into them.
	///
    void emitBuffers(int n, float birthTime, double* pos, double* vel, double* timestep, double* rgb, int rgbN, double* opacity);

    /// @brief  Emit from a jittered box.
	/// @detail The buffers must exist for us to emit into them.
	///
    void emitBox(const mat44f& xform, float spacing, float jitter, vec4f info, float birthTime, float birthTimeJitter=0);

    void emitContainer(ParticleContainer* masterContainer, float birthTime, float birthTimeJitter);

    void setExportedBuffers(const StringArray& exportRequests);

    void updateExportBuffers(Parameters& attributes);

public:

    ParticleBufferSpecs pendingBufferSpecs;
    ParticleBufferSpecs bufferSpecs;

    unsigned long long requestedUpdateState, _currentUpdateState;

	/// @brief  This mutex is released when processing is finished on this container.
    Mutex readyLock;

    //int pendingEmissionCount; // count of pending emission
    //int lastEmissionIndex; // counter for per-frame emission

    int numParticles; // number of current particles.
	int lastNumParticles;
    vec3f boundsLow;
    vec3f boundsHigh;

    bool enableSimulation;

    bool initialized;
    bool _firstTime;

    bool dirty;

    int maxParticles;

	StringArray _exportRequests;

    void _init();

    void _resetBuffers();

    ParticleBufferPtr _createBuffer(const ParticleBufferSpec& spec);

    friend class ParticleSolver;
    friend class ParticleModifier;
};

//------------------------------------------------------------------------------------------
}
}

#endif // NVPARTICLES_PARTICLE_CONTAINER_H_INCLUDED
