#ifndef NVPARTICLES_PARTICLE_MODIFIER_H_INCLUDED
#define NVPARTICLES_PARTICLE_MODIFIER_H_INCLUDED

//#pragma warning(push)
//#pragma warning(disable:4251)

#include "gl_utils.h"
#include "math_utils.h"
#include "CudaScheduler.h"

#include "NvParticlesExports.h"
#include "NvParticlesTypes.h"
#include "NvParticlesParticleBuffer.h"
#include "NvParticlesPrimitives.h"
#include "NvParticlesForces.h"
#include "NvParticlesParticleSolverImpl.h"

namespace Easy
{
namespace Cu
{
class Stream;
}

//------------------------------------------------------------------------------------------
namespace NvParticles
{
class Primitives;
class ParticleGrid;
class ParticleSolverImpl;
class ParticleRenderer;
class ParticleForces;
class ParticleContainer;
class PrimitiveResidentData;

//------------------------------------------------------------------------------------------
/// The simulator for the particles.
///
class _NvParticlesExport ParticleModifier : public HasParticleBuffers
{
    friend class ParticleContainer;


public:
    static int debugLevel;

    bool dirty; // requires a sync as data is pending
	unsigned long long requestedUpdateState, _currentUpdateState;

	bool isSolving; // is the solver in flight?
    // the assigned cuda device.
    int deviceIndex;

    float _currentTime;

    GLsync _glRenderSync[2];
    Mutex _renderLock;
    int _renderSyncIndex;
    int _renderNumParticles[2];

    Parameters attributes;

    SharedPtr<ParticleSolverImpl> solver;

    ParticleForces* particleForces;
    Primitives* primitives;
    PrimitiveResidentData* primitiveResidentData;

    ParticleGrid* particleGrid;

    Easy::Cu::Stream* cuStream;

	// solver method:
    std::string solverMethod;
    std::string pendingSolverMethod;

	// max particle count:
    int maxParticles;
    int pendingMaxParticles;

	// assigned container:
	ParticleContainer* container;

    bool enableSorting;
	int numParticles;

    bool initialized;
	int updateFlags;

    bool _ensureBuffers(const std::string& prefix, const ParticleBufferSpecs& bufferSpecs, bool clearFirst=false);

    ParticleBufferPtr _createBuffer(const ParticleBufferSpec& spec, int numItems, bool vbo=false);


	bool addBuffer(const ParticleBufferSpec& spec);

    /// @brief Dump the buffers.
	///
    void dump(size_t count=size_t(-1), size_t step=1);

    void calculateBoundsAsync();
	void computeBounds(vec3f& minCorner, vec3f& maxCorner);

    void setParameters(const Parameters& params);

    /// @brief Create all the context data.
    ///
    bool initialize();

	/// @brief Destroy all the context data.
	///
    void destroy();

	/// @brief Set the solver type.
	///
    void setSolverType(std::string type);

    /// @brief Set the maximum particles.
	///
    void setMaxParticles(int n);

    void setContainer(ParticleContainer* c);

    /// @brief Zero the buffers.
    ///
	void resetBuffers();

    void reset();
    void updateAsync();

    /// @brief		Read buffers from a container.
	/// @param		container		The source container.
	/// @param		offset			The source offset.
	/// @param		count			The source count.
	/// @warning	The container's buffers will be read asynchronously,
	///				so be careful about deleting them before a sync.
	///
    size_t readContainerAsync(ParticleContainer* container, size_t d_offset=0, size_t s_offset=0, size_t count=size_t(-1), bool useIncludes=false, std::string includes="", bool useExcludes=false, std::string excludes="");

	/// @brief		Write buffers to a container.
	/// @param		container		The source container.
	///
	size_t writeContainerAsync(ParticleContainer* container, bool storeBodyOnHost);

    size_t emitContainerAsync(ParticleContainer* container, bool useIncludes=false, std::string includes="", bool useExcludes=false, std::string excludes="");

	/// @brief		Copy data from this solver to the attached container.
	/// @warning	This will cause a sync.
	///
	void syncContainer();

	/// Update attributes with references to buffers and dynamic attributes.
    void updateRenderBuffers(NvParticles::Parameters& attributes, NvParticles::Parameters& buffers);

private:
    ParticleModifier(int cudaDevice);
    ~ParticleModifier();

    void _init();

	bool dirtyBounds;

    // buffers for transfering counts...
    Cu::Buffer h_firstDeadIndexBuffer;
    Cu::Buffer d_firstDeadIndexBuffer;

    // buffers for transfering bounds...
    Cu::Buffer d_posMinBuffer;
    Cu::Buffer d_posMaxBuffer;
    Cu::Buffer h_posMinBuffer;
    Cu::Buffer h_posMaxBuffer;

    // buffer for bounds reduction.
    Cu::Buffer reduceTempBuffer;

    friend class ParticleSolver;
};

//------------------------------------------------------------------------------------------
class _NvParticlesExport ParticleSolver
{
public:
    ParticleSolver();
    ~ParticleSolver();

    void destroy();
    bool setCudaDeviceConfig(int cudaDevice, void* display=0, void* glcontext=0);
    void updateAsync(bool solve=true, bool force=false);
    void sync();
    void setContainer(ParticleContainer* container);

    void setDebugLevel(int level);

	/// Set the time.
	///
    void setTime(float t);

	/// Set the solver type.
	///
    void setSolverType(std::string type);

    /// Set the solver parameters.
    ///
    void setParameters(const Parameters& params);

	/// Ensure a force exists.
	///
    void addForce(const std::string& name, const ForceData& item);

	/// Ensure a primitive exists.
	///
	void addPrimitive(const std::string& name, const Primitive& item);

    /// Render the solved particle data.
    /// This will block until the data is ready to be consumed.
    ///
    void render(ParticleRenderer* particleRenderer, bool drawBounds=false, bool drawGrid=false, bool drawPrimitives=false);

    void dump(size_t count=size_t(-1), size_t step=1);

    float getParticleSpacing();

    void updateContext();

public:
    ParticleModifier* _context;
    int _cudaDevice;
    void* _display;
    void* _glcontext;
    CudaScheduler* _cudaScheduler;
    ParticleContainer* _container;
    int _debugLevel;
    bool _isUpdating;
    std::string _solverMethod;
    ParticleForces* _forces;
    Primitives* _primitives;
    float _currentTime;
    Parameters _parameters;

    unsigned long long _currentUpdateState, _pendingUpdateState;
};

//------------------------------------------------------------------------------------------
}
}

//#pragma warning(pop)

#endif // NVPARTICLES_PARTICLE_MODIFIER_H_INCLUDED

