/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#include "NvParticlesParticleRenderer.h"
#include "NvParticlesManager.h"
#include "math_utils.h"

namespace Easy
{
namespace NvParticles
{

//------------------------------------------------------------------------------------------
ParticleRenderer::ParticleRenderer()
{
    _init();
    _impl = 0;
    _bgTexId = 0;
    _particleTexId = 0;
    _depthTexId = 0;
}

//------------------------------------------------------------------------------------------
void ParticleRenderer::_init()
{
    _window_w = 0;
    _window_h = 0;
    _type = "none";
    _initialized = false;
}

//------------------------------------------------------------------------------------------
ParticleRenderer::~ParticleRenderer()
{
    _destroy();
}

//------------------------------------------------------------------------------------------
void ParticleRenderer::updateParameters(const Parameters& parameters)
{
    _ensureInit();

    if (!_impl)
        return;

    _attributes.append(parameters);
	_attributes.setSpecs("renderer_", &_impl->parameterDefinitions());
    _impl->updateParameters(_attributes);
}

//------------------------------------------------------------------------------------------
void ParticleRenderer::render()
{
    if (!_impl)
        return;

	mat44f modelViewMat = _attributes.asMatrix44("modelViewMatrix");
	mat44f projMat = _attributes.asMatrix44("projectionMatrix");

	glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadMatrixf((GLfloat*)&modelViewMat);
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadMatrixf((GLfloat*)&projMat);

    //updateParameters();
    _impl->render();

	glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
    glCheckErrors();
}

//------------------------------------------------------------------------------------------
bool ParticleRenderer::setType(const std::string& type)
{
    if (_type == type)
        return true;

    _destroy();

    bool rc = true;
	_impl = Manager::getSingleton().createRenderer(type);
	if (!_impl)
	{
		return false;
	}

    _impl->_owner = this;
    _initialized = false;
    _type = type;
    return rc;
}

//------------------------------------------------------------------------------------------
void ParticleRenderer::_destroy()
{
    delete _impl;
    _impl = 0;

    if (_bgTexId)
        glDeleteTextures(1,&_bgTexId);

    if (_particleTexId)
        glDeleteTextures(1,&_particleTexId);

    if (_depthTexId)
        glDeleteTextures(1, &_depthTexId);

    _bgTexId = 0;
    _particleTexId = 0;
    _depthTexId = 0;

    _init();
}

//------------------------------------------------------------------------------------------
bool ParticleRenderer::valid()
{
    return (_impl != 0);
}

//------------------------------------------------------------------------------------------
void ParticleRenderer::_ensureInit()
{
    if (_initialized)
        return;
    _initialized = true;

    if (!_impl)
        return;

    _impl->initialize();

    // reload the current viewport attributes.
    float fov = _attributes.asFloat("viewFOV", 60*3.1416/180);
    int w = _attributes.asInt("viewWidth",800);
    int h = _attributes.asInt("viewHeight",800);

	_firstTime = true;
    resize(w, h, fov);
	_firstTime = false;

    int n = 0;
    _impl->_numParticles = n;
    _impl->_numComponents = 4;
    _impl->_posVbo = 0;
    _impl->_velVbo = 0;
    _impl->_colorVbo = 0;
}

//------------------------------------------------------------------------------------------
void ParticleRenderer::resize(int w, int h, float fov)
{
    if (!_impl)
        return;

    _ensureInit();

    //int oldW = _attributes.asInt("viewWidth", w);
    //int oldH = _attributes.asInt("viewHeight", h);

    FORCE_MAX(w,256);
    FORCE_MAX(h,256);

	bool isResized = true;

    if(_bgTexId && _depthTexId &&
        _window_w == w && _window_h == h)
    {
		isResized = false;
    }

    if (isResized)
    {
        _window_w = w;
        _window_h = h;
        _fov = fov;
        _aspect = (float)_window_w / (float)_window_h;
        _invFocalLen = (float)tanf(fov*0.5);

        // resize the buffers
        _createBuffers(w, h);
    }

	if (isResized || _firstTime)
	{
        _impl->_window_w = w;
        _impl->_window_h = h;
        _impl->_fov = fov;
        _impl->_aspect = (float) w / (float) h;
        _impl->_invFocalLen = (float)tanf(fov*0.5);
		_impl->resize(w, h, fov);
	}

    // we need to preserve the current viewport attributes.
    _attributes["viewFOV"] = fov;
    _attributes["viewWidth"] = w;
    _attributes["viewHeight"] = h;
}

//------------------------------------------------------------------------------------------
void ParticleRenderer::update(const Parameters& bufferParams)
{
    if (!_impl)
        return;

    _ensureInit();

    int n = bufferParams.asInt("numParticles",0);
    _impl->_numParticles = n;
    _impl->_numComponents = 4;
    _impl->_posVbo = bufferParams.asInt("position-vbo", 0);
    _impl->_velVbo = bufferParams.asInt("velocity-vbo", 0);
    _impl->_colorVbo = bufferParams.asInt("color-vbo", 0);
    _impl->update(_attributes, bufferParams);
}

//------------------------------------------------------------------------------------------
void ParticleRenderer::renderDebugTextures(int x, int y, float scale)
{
    if (!_impl)
        return;

    glActiveTexture(GL_TEXTURE0);

    int sx = x;
    int sw = int(_window_w*scale);
    int sh = int(_window_h*scale);

    if (_depthTexId)
    {
        int tw = int(_window_w);
        int th = int(_window_h);
        glEnable(GL_TEXTURE_RECTANGLE);

        glBindTexture(GL_TEXTURE_RECTANGLE, _depthTexId);
        glBegin(GL_QUADS);
        glTexCoord2f(0,0);
        glVertex3f( sx+0, 0,0);
        glTexCoord2f(tw,0);
        glVertex3f(sx+sw, 0,0);
        glTexCoord2f(tw,th);
        glVertex3f(sx+sw, sh,0);
        glTexCoord2f(0,th);
        glVertex3f( sx+0, sh,0);
        glEnd();
        glBindTexture(GL_TEXTURE_RECTANGLE, 0);
        sx += sw;
    }

    if (_bgTexId)
    {
        int tw = int(_window_w);
        int th = int(_window_h);
        glEnable(GL_TEXTURE_RECTANGLE);

        glBindTexture(GL_TEXTURE_RECTANGLE, _bgTexId);
        glBegin(GL_QUADS);
        glTexCoord2f(0,0);
        glVertex3f( sx+0, 0,0);
        glTexCoord2f(tw,0);
        glVertex3f(sx+sw, 0,0);
        glTexCoord2f(tw,th);
        glVertex3f(sx+sw, sh,0);
        glTexCoord2f(0,th);
        glVertex3f( sx+0, sh,0);
        glEnd();
        glBindTexture(GL_TEXTURE_RECTANGLE, 0);
        sx += sw;
    }

    glDisable(GL_TEXTURE_RECTANGLE);
    glCheckErrors();

    _impl->renderDebugTextures(sx, y, scale);
}

//------------------------------------------------------------------------------------------
void ParticleRenderer::_createBuffers(int w, int h)
{
    if (_bgTexId == 0)
        glGenTextures(1, &_bgTexId);

    glBindTexture(GL_TEXTURE_RECTANGLE, _bgTexId);
    glTexImage2D(GL_TEXTURE_RECTANGLE, 0, GL_RGBA, w, h, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL);
    glCheckErrors();
    glTexParameterf(GL_TEXTURE_RECTANGLE, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_RECTANGLE, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glBindTexture(GL_TEXTURE_RECTANGLE, 0);
    glCheckErrors();

    if (_depthTexId == 0)
        glGenTextures(1, &_depthTexId);

    glBindTexture(GL_TEXTURE_RECTANGLE, _depthTexId);
    glTexImage2D(GL_TEXTURE_RECTANGLE, 0, GL_DEPTH_COMPONENT, w, h, 0, GL_DEPTH_COMPONENT, GL_UNSIGNED_INT, NULL);
    glCheckErrors();
    glTexParameterf(GL_TEXTURE_RECTANGLE, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameterf(GL_TEXTURE_RECTANGLE, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glBindTexture(GL_TEXTURE_RECTANGLE, 0);
    glCheckErrors();
}

//------------------------------------------------------------------------------------------
void ParticleRenderer::readColorTexture()
{
    glBindTexture(GL_TEXTURE_RECTANGLE, _bgTexId);
    glCopyTexImage2D( GL_TEXTURE_RECTANGLE,
                      0,
                      GL_RGBA,
                      0,
                      0,
                      _window_w,
                      _window_h,
                      0 );
    glBindTexture(GL_TEXTURE_RECTANGLE, 0);
    glCheckErrors();
}

//------------------------------------------------------------------------------------------
void ParticleRenderer::readDepthTexture()
{
    // does this work?
    glBindTexture(GL_TEXTURE_RECTANGLE, _depthTexId);
    glCopyTexImage2D( GL_TEXTURE_RECTANGLE,
                      0,
                      GL_DEPTH_COMPONENT,
                      0,
                      0,
                      _window_w,
                      _window_h,
                      0 );
    glBindTexture(GL_TEXTURE_RECTANGLE, 0);
    glCheckErrors();
}

//------------------------------------------------------------------------------------------
bool ParticleRenderer::bindDepthTexture(GLenum texTarget)
{
    if (!_depthTexId)
        return false;
    glCheckErrors();
    glBindTexture(texTarget, _depthTexId);
    if (!glCheckErrors())
    {
        return false;
    }
    return true;
}

//------------------------------------------------------------------------------------------
ParticleRendererImpl::ParticleRendererImpl()
    :
    _window_w(0), _window_h(0), // this ensures we create buffers on init
    _numParticles(0),
    _posVbo(0),
    _velVbo(0),
    _colorVbo(0),
    _numComponents(0),
    _indexBuffer(0)
{
}

//------------------------------------------------------------------------------------------
ParticleRendererImpl::~ParticleRendererImpl()
{
}

//------------------------------------------------------------------------------------------
void ParticleRendererImpl::_draw(GLenum type, int start, int count, bool indexed)
{
    if(count == -1)
        count = _numParticles;

    // get local count in batch
    int batchStart = start;
    if(batchStart > _numParticles)
    {
        start -= _numParticles;
        return;
    }

    int batchCount = std::min(count, _numParticles - batchStart);
    if(batchCount == 0)
    {
        return;
    }

    count -= batchCount;
    start = 0;

    if (_posVbo)
    {
        glBindBuffer(GL_ARRAY_BUFFER, _posVbo);
        glVertexPointer(_numComponents, GL_FLOAT, 0, 0);
        glEnableClientState(GL_VERTEX_ARRAY);
    }

    if (_colorVbo)
    {
        glBindBuffer(GL_ARRAY_BUFFER, _colorVbo);
        glColorPointer(4, GL_FLOAT, 0, 0);
        glEnableClientState(GL_COLOR_ARRAY);
    }
    else
    {
        // just in case this was set somewhere else!
        glDisableClientState(GL_COLOR_ARRAY);
    }

    if (_velVbo)
    {
        glBindBuffer(GL_ARRAY_BUFFER, _velVbo);
        glClientActiveTexture(GL_TEXTURE1);
        glTexCoordPointer(4, GL_FLOAT, 0, 0);
        glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    }

    if(indexed && _indexBuffer)
    {
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, _indexBuffer);
        glDrawElements(type, batchCount, GL_UNSIGNED_INT, (void*)(batchStart*sizeof(unsigned int)));
    }
    else
    {
        glDrawArrays(type, batchStart, batchCount);
    }

    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    glDisableClientState(GL_VERTEX_ARRAY);
    glDisableClientState(GL_COLOR_ARRAY);
    glClientActiveTexture(GL_TEXTURE1);
    glDisableClientState(GL_TEXTURE_COORD_ARRAY);
    glClientActiveTexture(GL_TEXTURE0);
    glDisableClientState(GL_TEXTURE_COORD_ARRAY);

    glCheckErrors();
}

//------------------------------------------------------------------------------------------
void ParticleRendererImpl::update(const NvParticles::Parameters& attributes, const NvParticles::Parameters& buffers)
{
}

//------------------------------------------------------------------------------------------
void ParticleRendererImpl::resize(int w, int h, float fov)
{
}

//------------------------------------------------------------------------------------------
void ParticleRendererImpl::initialize()
{
}

//------------------------------------------------------------------------------------------
void ParticleRendererImpl::updateParameters(Parameters& params)
{
}

//------------------------------------------------------------------------------------------
void ParticleRendererImpl::renderDebugTextures(int x, int y, float scale)
{
}

//------------------------------------------------------------------------------------------
}
}
