/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#include "NvParticlesParticleRendererImpl_Point.h"
#include "std_utils.h"

#define STRINGIFY(A) #A

namespace Easy
{
namespace NvParticles
{

//------------------------------------------------------------------------------------------
static const char *pointVertexShaderColor = STRINGIFY(
    uniform vec4 color;
    uniform float colorScale;
    varying vec4 velocity;
            void main()
{
    velocity = gl_MultiTexCoord1;
    gl_Position = gl_ModelViewProjectionMatrix * vec4(gl_Vertex.xyz, 1.0);
    gl_FrontColor = vec4(color.rgb*colorScale, color.a);
}
        );

//------------------------------------------------------------------------------------------
static const char *pointVertexShader = 
    STRINGIFY(
    uniform float colorScale;
    varying vec4 velocity;
            void main()
{
    velocity = gl_MultiTexCoord1;
    gl_Position = gl_ModelViewProjectionMatrix * vec4(gl_Vertex.xyz, 1.0);
    gl_FrontColor = vec4(gl_Color.rgb*colorScale, gl_Color.a);
}
        );

//------------------------------------------------------------------------------------------
static const char *pointPixelShader = 
"#version 120\n"    
    STRINGIFY(

varying vec4 velocity;

void main()
{
    gl_FragColor = gl_Color;
}
        );

//------------------------------------------------------------------------------------------
static const char *streakVertexShader =
"#version 140\n"
"#extension GL_ARB_compatibility : enable\n"
STRINGIFY(
	out vec3 oPos;
	uniform mat4 modelViewProjMat;
void main()
{
	oPos = gl_Vertex.xyz;
	gl_Position = modelViewProjMat * vec4(oPos, 1.0);
    gl_FrontColor = gl_Color;
    gl_BackColor = gl_Color;
    gl_TexCoord[0].x = float(gl_VertexID);
	gl_TexCoord[1] = gl_MultiTexCoord1;
}
        );

//------------------------------------------------------------------------------------------
static const char *streakGeometryShader =
"#version 150\n"
"#extension GL_EXT_geometry_shader4 : enable\n"
"layout(points) in;\n"
"layout(line_strip, max_vertices = 2) out;\n"
STRINGIFY(
	out float gAlpha;\n
	in vec3 oPos[1];\n
	uniform mat4 modelViewProjMat;\n
	uniform float streakLength;\n
void main()\n
{
    gl_PrimitiveID = int(gl_TexCoordIn[0][0].x);
	gl_FrontColor = gl_FrontColorIn[0];\n
	vec3 oVel = gl_TexCoordIn[0][1].xyz * streakLength;
	if(dot(oVel,oVel) < 0.1)
		oVel = (oVel*0.1) / length(oVel);

	gAlpha = smoothstep(1,0,length(oVel)/2);

	// tail...
	gAlpha = 0;\n
    gl_Position = modelViewProjMat * vec4(oPos[0]-oVel, 1);\n
    EmitVertex();\n

	// head...
	gAlpha = 1;
    gl_Position = gl_in[0].gl_Position;\n
    EmitVertex();\n
}
        );

//------------------------------------------------------------------------------------------
static const char *streakPixelShader = STRINGIFY(

in float gAlpha;

void main()
{
    gl_FragColor = vec4(gl_Color.xyz, gAlpha);
}
        );

//------------------------------------------------------------------------------------------
ParticleRendererImpl_Point::ParticleRendererImpl_Point()
    :
    _pointSize(5.0f)
    ,_colorScale(1)
    ,_pointProgram(0)
    ,_pointProgramColor(0)
	,_streakProgram(0)
    ,_useColor(false)
    ,_color(make_vec4f(1))
	,_streakLength(1)
	,_streakWidth(1)
{
	defineParameter("__description", ParameterSpec::STRING, std::string("Simple point renderer"));
	defineParameter("pointSize", ParameterSpec::INT, int(2), Parameters().set("description", std::string("Simple point renderer")));
	defineParameter("useColor", ParameterSpec::BOOL, bool(false), Parameters().set("description", std::string("Use the custom color rather than the particle-color")));
	defineParameter("color", ParameterSpec::VEC4,  make_vec4f(1,1,0,1), Parameters().set("description", std::string("Custom color")));
	defineParameter("streakLength", ParameterSpec::FLOAT, float(0), Parameters().set("description", std::string("How much to streak along the motion vector")));
	defineParameter("streakWidth", ParameterSpec::FLOAT, float(1), Parameters().set("description", std::string("Width of the streak")));
}

//------------------------------------------------------------------------------------------
void* ParticleRendererImpl_Point::creator()
{
	return new ParticleRendererImpl_Point();
}

//------------------------------------------------------------------------------------------
void ParticleRendererImpl_Point::initialize()
{
    _pointProgram = gl::compileProgram(pointVertexShader, pointPixelShader);
    if (!_pointProgram)
        abort();
    _pointProgramColor = gl::compileProgram(pointVertexShaderColor, pointPixelShader);
    if (!_pointProgramColor)
        abort();

    _streakProgram = gl::compileProgram(streakVertexShader, streakPixelShader, streakGeometryShader);
    if (!_streakProgram)
        abort();

}

//------------------------------------------------------------------------------------------
void ParticleRendererImpl_Point::updateParameters(Parameters& params)
{
    _pointSize = params.inputValue(getParameter("pointSize"), _pointSize)->asInt();
    _colorScale = params.inputValue(getParameter("colorScale"), _colorScale)->asFloat();
    _useColor = params.inputValue(getParameter("useColor"), _useColor)->asBool();

    _streakLength = params.inputValue(getParameter("streakLength"), _streakLength)->asFloat();
	_streakWidth = params.inputValue(getParameter("streakWidth"), _streakWidth)->asFloat();

	vec4f v;
    v = params.inputValue(getParameter("color"), vec4f::fromArray((float*)&_color))->asVector4();
    _color = vec4f::fromArray((float*)&v);
}

//------------------------------------------------------------------------------------------
void ParticleRendererImpl_Point::render()
{
    glDepthMask(GL_TRUE);
    glEnable(GL_DEPTH_TEST);

	if (_streakLength != 0)
	{
		mat44f modelViewMat, projMat, m;
        glGetFloatv(GL_MODELVIEW_MATRIX, (GLfloat*)&modelViewMat);
        glGetFloatv(GL_PROJECTION_MATRIX, (GLfloat*)&projMat);
		m = projMat * modelViewMat;

		glUseProgram(_streakProgram);
		glUniform1f(glGetUniformLocation(_streakProgram, "streakLength"), _streakLength);
		glUniformMatrix4fv(glGetUniformLocation(_streakProgram, "modelViewProjMat"), 1, false, (GLfloat*)&m);
		//glUniform1f(glGetUniformLocation(pointProgram, "colorScale"), colorScale);

	    glPointSize(1);

		glLineWidth(std::max(1,(int)_streakWidth));

		glEnable(GL_BLEND);
	}
	else
	{
		_colorScale = 1.f;
		if (_useColor)
		{
			glUseProgram(_pointProgramColor);
			glUniform1f( glGetUniformLocation(_pointProgramColor, "colorScale"), _colorScale);
			glUniform4fv( glGetUniformLocation(_pointProgramColor, "color"), 4, (GLfloat*)&_color);
		}
		else
		{
			glUseProgram(_pointProgram);
			glUniform1f( glGetUniformLocation(_pointProgram, "colorScale"), _colorScale);
		}

		if (_pointSize < 1)
            _pointSize = 1;
		glPointSize(_pointSize);
	}

	glColor4f(1,1,1,1);

	int oldColorVbo;
	if (_useColor)
	{
		// dangerous, but it works...
		oldColorVbo = _colorVbo;
		_colorVbo = 0;
	}

    _draw(GL_POINTS);

	if (_useColor)
	{
		_colorVbo = oldColorVbo;
	}

	glDisable(GL_BLEND);
	glLineWidth(1);
	glPointSize(1);
    glUseProgram(0);
}

//------------------------------------------------------------------------------------------
}
}
