/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#include "NvParticlesParticleRendererImpl_Sphere.h"
#include "std_utils.h"

#define STRINGIFY(A) #A

namespace Easy
{
namespace NvParticles
{

//------------------------------------------------------------------------------------------
static const char *sphere_vertex_shader = STRINGIFY(
            uniform float pointRadius;  // point size in world space
            uniform float pointScale;   // scale to calculate size in pixels
            uniform float colorScale;
			varying vec3 eLightDir;
            void main()
{
    // calculate window-space point size
    vec3 ePos = vec3(gl_ModelViewMatrix * vec4(gl_Vertex.xyz, 1.0));
    float dist = length(ePos);
    gl_Position = gl_ModelViewProjectionMatrix * vec4(gl_Vertex.xyz, 1.0);

	eLightDir = normalize(vec3(gl_NormalMatrix * vec3(0,1,0)));

    // use the last channel of the float4 as a scaling factor.
    float pointRez = gl_Vertex.w;
    if (pointRez == 0)
        pointRez = 1000000;
    gl_PointSize = (1/pointRez) * pointRadius * (pointScale / dist);

    gl_TexCoord[0] = gl_MultiTexCoord0;
    gl_TexCoord[1] = gl_MultiTexCoord1;
    gl_TexCoord[2] = gl_MultiTexCoord2;
    gl_FrontColor = vec4(gl_Color.rgb*colorScale, gl_Color.a);

	gl_TexCoord[3] = vec4(ePos.xyz, 1);
}
        );

//------------------------------------------------------------------------------------------
static const char *shadedSphereFragmentShader = STRINGIFY(
            uniform float colorScale;
			in vec3 eLightDir;
            void main()
{
    // calculate normal from texture coordinates
    vec3 N;
    N.xy = gl_TexCoord[0].xy*vec2(2.0, -2.0) + vec2(-1.0, 1.0);
    float mag = dot(N.xy, N.xy);
    if (mag > 1.0)
        discard;   // kill pixels outside circle

    N.z = sqrt(1.0-mag);

	// calculate depth
	const float depthBias = 0.0;
    vec4 ePos = vec4(gl_TexCoord[3].xyz + N*gl_TexCoord[3].w, 1.0);   // position of this pixel on sphere in eye space
    vec4 cPos = gl_ProjectionMatrix * (ePos + vec4(0.0, 0.0, depthBias, 0.0));
    gl_FragDepth = (cPos.z / cPos.w)*0.5+0.5;

    // calculate simple lighting
	// float diffuse = max(0.0, dot(eLightDir, N));
	const float wrap = 0.75;
	float wrapDiffuse = max(0.0, (dot(eLightDir, N) + wrap) / (1.0 + wrap));

    vec4 final = gl_Color * wrapDiffuse;

    //float specular = pow(max(0.0, dot(reflect(eLightDir, N), vec3(0.0,0.0,-1.0))), 20.0);
    //final += specular * vec4(1.0);

    gl_FragColor = vec4(final.rgb * colorScale, gl_Color.a);
}
        );

//------------------------------------------------------------------------------------------
static const char *unshadedSphereFragmentShader = STRINGIFY(
            uniform float colorScale;
            void main()
{
    // calculate normal from texture coordinates
    vec3 N;
	N.xy = gl_TexCoord[0].xy*vec2(2.0, -2.0) + vec2(-1.0, 1.0);
	float mag = dot(N.xy, N.xy);
    if (mag > 1.0)
        discard;   // kill pixels outside circle

	gl_FragColor = vec4(gl_Color.rgb*colorScale, gl_Color.a);

    N.z = sqrt(1.0-mag);
    // calculate depth
	const float depthBias = 0;
    vec4 ePos = vec4(gl_TexCoord[3].xyz + N*gl_TexCoord[3].w, 1.0);   // position of this pixel on sphere in eye space
    vec4 cPos = gl_ProjectionMatrix * (ePos + vec4(0, 0, depthBias, 0));
    gl_FragDepth = (cPos.z / cPos.w)*0.5+0.5;
}
        );

//------------------------------------------------------------------------------------------
static const char *streak_fragment_shader = STRINGIFY(
            void main()
{
    gl_FragColor = gl_Color;
}
        );

//------------------------------------------------------------------------------------------
static const char *streak_vertex_shader =
"#version 140\n"
"#extension GL_ARB_compatibility : enable\n"
STRINGIFY(
            void main()
{
    gl_Position = ftransform();
    gl_FrontColor = gl_Color;
    gl_BackColor = gl_Color;
    gl_TexCoord[0].x = float(gl_VertexID);
}
        );

//------------------------------------------------------------------------------------------
static const char *streak_geometry_shader =
"#version 120\n"
"#extension GL_EXT_geometry_shader4 : enable\n"
STRINGIFY(
            void main()
{
    int i;

    gl_PrimitiveID = int(gl_TexCoordIn[0][0].x);

    vec3 pos = gl_PositionIn[0].xyz;                           \n

    gl_Position = gl_ProjectionMatrix * vec4(pos, 1); \n
    EmitVertex();                                              \n

}
        );

//------------------------------------------------------------------------------------------
ParticleRendererImpl_Sphere::ParticleRendererImpl_Sphere()
    :
    _useLighting(true)
    ,_particleRadius(0.125f * 0.5f)
    ,_colorScale(1)
	,_useColor(false)
	,_color(make_vec4f(1))
    ,_shadedSphereProgram(0)
    ,_unshadedSphereProgram(0)
    ,_unshadedSphereMblurProgram(0)
{
    defineParameter("__description", ParameterSpec::STRING, std::string("Simple sphere renderer"));
    defineParameter("absoluteRadius", ParameterSpec::FLOAT, 0.f, Parameters().set("description", "Particle absolute radius (set to zero to use default)"));
    defineParameter("radiusFactor", ParameterSpec::FLOAT, 1.f, Parameters().set("description", "Particle radius multiplier"));
	defineParameter("useColor", ParameterSpec::BOOL, false, Parameters().set("description", "Use the custom color rather than the particle-color"));
	defineParameter("color", ParameterSpec::VEC4, make_vec4f(1,1,0,1), Parameters().set("description", "Custom color"));
	defineParameter("useLighting", ParameterSpec::BOOL, false, Parameters().set("description", "Shade the spheres"));
}

//------------------------------------------------------------------------------------------
void* ParticleRendererImpl_Sphere::creator()
{
	return new ParticleRendererImpl_Sphere();
}

//------------------------------------------------------------------------------------------
void ParticleRendererImpl_Sphere::initialize()
{
    _shadedSphereProgram = gl::compileProgram(sphere_vertex_shader, shadedSphereFragmentShader);
    if(!_shadedSphereProgram)
        abort();

    _unshadedSphereProgram = gl::compileProgram(sphere_vertex_shader, unshadedSphereFragmentShader);
    if(!_unshadedSphereProgram)
        abort();

    int num_vertices = 2;
    glGetIntegerv(GL_MAX_GEOMETRY_OUTPUT_VERTICES_EXT,&num_vertices);
    _unshadedSphereMblurProgram = gl::compileProgram(streak_vertex_shader, streak_fragment_shader, streak_geometry_shader, GL_POINTS, GL_LINE_STRIP, 2);
    if(!_unshadedSphereMblurProgram)
        abort();

#if !defined(__APPLE__) && !defined(MACOSX)
    glClampColorARB(GL_CLAMP_VERTEX_COLOR_ARB, GL_FALSE);
    glClampColorARB(GL_CLAMP_FRAGMENT_COLOR_ARB, GL_FALSE);
#endif
}

//------------------------------------------------------------------------------------------
void ParticleRendererImpl_Sphere::updateParameters(Parameters& params)
{
    float pr = params.inputValue(getParameter("absoluteRadius"), 0)->asFloat();
    if(pr == 0)
        pr = params.inputValue(getParameter("radiusFactor"), 1)->asFloat() * params.asFloat("__particleRadius", 0.125f * 0.5f);
    _particleRadius = pr;
    _colorScale = params.inputValue(getParameter("colorScale"), _colorScale)->asFloat();
    _useLighting = params.inputValue(getParameter("useLighting"), _useLighting)->asBool();
    _useColor = params.inputValue(getParameter("useColor"), _useColor)->asBool();
	vec4f v;
    v = params.inputValue(getParameter("color"), vec4f::fromArray((float*)&_color))->asVector4();
    _color = vec4f::fromArray((float*)&v);
}

//------------------------------------------------------------------------------------------
void ParticleRendererImpl_Sphere::render()
{
    glEnable(GL_POINT_SPRITE_ARB);
    glTexEnvi(GL_POINT_SPRITE_ARB, GL_COORD_REPLACE_ARB, GL_TRUE);
    glEnable(GL_VERTEX_PROGRAM_POINT_SIZE_NV);
    glDepthMask(GL_TRUE);
    glEnable(GL_DEPTH_TEST);

    if (_useLighting)
    {
        glUseProgram( _shadedSphereProgram );
        glUniform1f( glGetUniformLocation(_shadedSphereProgram, "pointScale"), _window_h / tanf(_fov*0.5f) );
        glUniform1f( glGetUniformLocation(_shadedSphereProgram, "pointRadius"), _particleRadius );
        glUniform1f( glGetUniformLocation(_shadedSphereProgram, "colorScale"), _colorScale );
    }
    else
    {
        glUseProgram(_unshadedSphereProgram);
        glUniform1f( glGetUniformLocation(_unshadedSphereProgram, "pointScale"), _window_h / tanf(_fov*0.5f) );
        glUniform1f( glGetUniformLocation(_unshadedSphereProgram, "pointRadius"), _particleRadius );
        glUniform1f( glGetUniformLocation(_unshadedSphereProgram, "colorScale"), _colorScale );
    }

	int oldColorVbo;
	if (_useColor)
	{
		// dangerous, but it works...
		oldColorVbo = _colorVbo;
		_colorVbo = 0;
	}

    _draw(GL_POINTS);

	if (_useColor)
	{
		_colorVbo = oldColorVbo;
	}

    glUseProgram(0);
    glDisable(GL_POINT_SPRITE_ARB);
	glDisable(GL_VERTEX_PROGRAM_POINT_SIZE_NV);
}
//------------------------------------------------------------------------------------------
}
}
