/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#include "gl_utils.h"

#include "NvParticlesConfig.h"
#include "NvParticlesProfiler.h"
#include "NvParticlesPrimitives.h"
#include "cuda_utils.h"
#include "cuda_std_utils.h"
#include "math_utils_cuda.h"

namespace Easy
{
namespace NvParticles
{

int Primitives::debugLevel = 0;

//------------------------------------------------------------------------------------------
Primitive Primitives::Invalid;

//------------------------------------------------------------------------------------------
PrimitiveResidentData::PrimitiveResidentData()
{
    d_transformData = new Cu::Buffer(Cu::Buffer::CUDA, sizeof(mat44f)*NVPARTICLES_PRIMITIVE_MAX_COUNT);
    h_transformData = new Cu::Buffer();
    h_transformData->Allocate(Cu::Buffer::HOST, sizeof(mat44f)*NVPARTICLES_PRIMITIVE_MAX_COUNT, Cu::Buffer::HOST_PINNED);
}

//------------------------------------------------------------------------------------------
PrimitiveResidentData::~PrimitiveResidentData()
{
}

//------------------------------------------------------------------------------------------
void PrimitiveResidentData::updateTransforms(class Primitives* owner, float t, long stream)
{
    Cu::Buffer::CopyOptions co = Cu::Buffer::CopyOptions().SetStream(stream);
    Cu::BufferMapper<mat44f> h_transformDataPtr(Cu::Buffer::HOST);

    h_transformDataPtr.Map(*h_transformData, Cu::Buffer::MapOptions().SetStream(stream));

    for (int i=0;i<NVPARTICLES_PRIMITIVE_MAX_COUNT;++i)
    {
        if (owner->group.primitives[i].type == Primitive::PRIMITIVE_NONE)
            continue;

        // get the interpolated matrix.
        mat44f xform = mat44f::slerp(t, owner->group.primitives[i].prevXform, owner->group.primitives[i].xform);

        h_transformDataPtr[i] = xform;

		// store the pointer offset.
        owner->group.transforms[i].xform = ((mat44f*)d_transformData->Data()) + i;
    }

    h_transformDataPtr.Unmap();

    d_transformData->Copy(*h_transformData, 0, 0, sizeof(mat44f)*NVPARTICLES_PRIMITIVE_MAX_COUNT, co);
}

//------------------------------------------------------------------------------------------
Primitives::Primitives()
{
    clear();
    group.internalScale = 1.0;
    group.restitution = 1.0;
    group.friction = 0.0;
}

//------------------------------------------------------------------------------------------
Primitive& Primitives::get(const std::string& name)
{
    for (int i=0;i<nextPrimitiveIndex;++i)
    {
        if (names[i] == name)
            return group.primitives[i];
    }
    return Invalid;
}

bool Primitives::remove(const std::string& name)
{
    Primitive& data = get(name);
    if (data.type == Primitive::PRIMITIVE_NONE)
        return false;

    if (debugLevel)
        printInfo("Removing primitive: " + name);

    group.transforms[data.transformIndex].xform = NULL;
    names[data.transformIndex] = "";
    memset((void*)&data, 0, sizeof(Primitive));
/*
    // compact the list of primitives...
    for(int i=0;i<nextPrimitiveIndex;++i)
    {
        if(group.primitives[i].type != Primitive::PRIMITIVE_NONE)
            continue;
        --nextPrimitiveIndex;
        group.primitives[i] = group.primitives[nextPrimitiveIndex];
        group.primitives[i].transformIndex = i;

        group.primitives[nextPrimitiveIndex].type = Primitive::PRIMITIVE_NONE;
        group.primitives[nextPrimitiveIndex].xform = mat44f::identity();
        group.primitives[nextPrimitiveIndex].imageIndex = 0;

        group.transforms[i] = primitives.transforms[nextPrimitiveIndex];
        group.transforms[nextPrimitiveIndex].xform = 0;
    }*/
    return true;
}

//------------------------------------------------------------------------------------------
bool Primitives::add(const std::string& name, const Primitive& f)
{
    bool rc = true;

    if (nextPrimitiveIndex == NVPARTICLES_PRIMITIVE_MAX_COUNT)
    {
        // compact the list of primitives...
        for(int i=0;i<nextPrimitiveIndex;++i)
        {
            if (group.primitives[i].type != Primitive::PRIMITIVE_NONE)
                continue;
            --nextPrimitiveIndex;

            group.primitives[i] = group.primitives[nextPrimitiveIndex];
            group.primitives[i].transformIndex = i;

            group.primitives[nextPrimitiveIndex].type = Primitive::PRIMITIVE_NONE;
            group.primitives[nextPrimitiveIndex].xform = mat44f::identity();

            group.transforms[i] = group.transforms[nextPrimitiveIndex];
            group.transforms[nextPrimitiveIndex].xform = 0;
        }
    }

    mat44f prevXform = f.prevXform;

    int useIndex = nextPrimitiveIndex;
    for(int i=0;i<nextPrimitiveIndex;++i)
    {
        if (names[i] == name)
        {
            assert(group.primitives[i].type != Primitive::PRIMITIVE_NONE);
            rc = false;
            useIndex = i;

            // if we are replacing the primitive with another of the same name,
            // then use the previous xform as the previous position!
            prevXform = group.primitives[useIndex].xform;

            if (debugLevel)
                printInfo("Updating primitive: " + name);
        }
    }

	if (useIndex >= NVPARTICLES_PRIMITIVE_MAX_COUNT)
    {
        printError(Stringf("Unable to create primitive: %s (more than %d primitive)", name.c_str(), NVPARTICLES_PRIMITIVE_MAX_COUNT));
		return false;
	}

    // add the new primitive.
    group.primitives[useIndex] = f;
    // set the internal attributes.
    group.primitives[useIndex].prevXform = prevXform;
    group.primitives[useIndex].transformIndex = useIndex;

    // we will fill this in later when the solver's evaluate method calls "updateTransforms".
    group.transforms[useIndex].xform = 0;

    names[useIndex] = name;
    needUpload = true;

    if (useIndex == nextPrimitiveIndex)
    {
        if (debugLevel)
            printInfo("Created primitive: " + name);
        nextPrimitiveIndex++;
    }

    group.numPrimitives = nextPrimitiveIndex;

    return rc;
}

//------------------------------------------------------------------------------------------
void Primitives::clear()
{
    // this is redundant!
    memset(&group, 0, sizeof(PrimitiveGroup));

    // remove all the primitives...
    for (int i=0;i<NVPARTICLES_PRIMITIVE_MAX_COUNT;++i)
    {
        group.primitives[i].type = Primitive::PRIMITIVE_NONE;
        group.primitives[i].transformIndex = i;
        group.primitives[i].xform = mat44f::identity();
    }

	/// CAVEAT:
	// the resident data is still allocated.
	// This must be removed while inside the thread (i.e. in the "Update" methods)

    // initialize the manager...
    nextPrimitiveIndex = 0;
    needUpload = true;

}

//------------------------------------------------------------------------------------------
bool Primitives::add(const std::string& name, int type, mat44f prevXform, mat44f xform, vec3f extents, int flags)
{
    Primitive f;

	f.transformIndex = 0;
    f.type = type;
    f.extents = extents;
    f.xform = xform;
    f.prevXform = prevXform;
    f.flags = flags;

    return add(name,f);
}

//------------------------------------------------------------------------------------------
void Primitives::drawSphere(const Primitive& p)
{
    vec3f extents = make_vec3f(p.extents);
    mat44f mat = p.xform;

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glMultMatrixf((GLfloat*)&mat);
    vec3f center = make_vec3f(0);
    gl::drawWireSphere((float*)&center, 1);
    glPopMatrix();
}

//------------------------------------------------------------------------------------------
void Primitives::drawBox(const Primitive& p)
{
    vec3f extents = make_vec3f(p.extents);
    mat44f mat = p.xform;

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glMultMatrixf((GLfloat*)&mat);
    gl::drawWireCube(1);
    glPopMatrix();
}

//------------------------------------------------------------------------------------------
void Primitives::drawCapsule(const Primitive& p)
{
    vec3f extents = make_vec3f(p.extents);
    mat44f mat = p.xform;

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glMultMatrixf((GLfloat*)&mat);

    vec3f otherPos = make_vec3f(0, 0, extents.z);
    vec3f zero = make_vec3f(0);

    glTranslatef(0, 0, -extents.z/2);
    gl::drawWireCylinder((float*)&zero, (float*)&otherPos, 1, 1, 8, false);
    gl::drawWireSphere((float*)&zero, 1, 16, 0, PI, 2*PI, -PI/2, PI/2);
    gl::drawWireSphere((float*)&otherPos, 1, 16, 0, 0, PI, -PI/2, PI/2);

    glPopMatrix();
}

//------------------------------------------------------------------------------------------
void Primitives::drawPlane(const Primitive& p)
{
    vec3f extents = make_vec3f(p.extents);
    mat44f mat = p.xform;

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glMultMatrixf((GLfloat*)&mat);

    float div;

    vec3f first = -extents/2;
    vec3f second = extents/2;

    glBegin(GL_LINES);

    div = (second.x) / 1;
    if (div == 0)
        div = 1;
    for (float x=first.x; x<=second.x; x+=div)
    {
        glVertex3f(x,first.y,0);
        glVertex3f(x,second.y,0);
    }

    div = (second.y) / 1;
    if (div == 0)
        div = 1;
    for (float y=first.y; y<=second.y; y+=div)
    {
        glVertex3f(first.x,y,0);
        glVertex3f(second.x,y,0);
    }

    glEnd();

    glPopMatrix();
}

//------------------------------------------------------------------------------------------
void Primitives::render()
{
    for (int i=0; i<NVPARTICLES_PRIMITIVE_MAX_COUNT; ++i)
    {
        if (group.primitives[i].type == Primitive::PRIMITIVE_SPHERE)
        {
            drawSphere(group.primitives[i]);
        }
        else if (group.primitives[i].type == Primitive::PRIMITIVE_BOX)
        {
            drawBox(group.primitives[i]);
        }
        else if (group.primitives[i].type == Primitive::PRIMITIVE_CAPSULE)
        {
            drawCapsule(group.primitives[i]);
        }
        else if (group.primitives[i].type == Primitive::PRIMITIVE_PLANE)
        {
            drawPlane(group.primitives[i]);
        }
    }
}

//------------------------------------------------------------------------------------------
}
}
