/*
 * Copyright 1993-2012 NVIDIA Corporation.  All rights reserved.
 *
 * Please refer to the NVIDIA end user license agreement (EULA) associated
 * with this source code for terms and conditions that govern your use of
 * this software. Any use, reproduction, disclosure, or distribution of
 * this software and related documentation outside the terms of the EULA
 * is strictly prohibited.
 *
 */

#ifndef WCSPH_KERNELS_H_INCLUDED
#define WCSPH_KERNELS_H_INCLUDED

#ifndef M_PI
#define M_PI	3.14159265358979323846f
#endif

namespace sph
{

class Wendland
{
public:
	static NVPARTICLES_CUDA_EXPORT
    float kernelConstant(const float h)
	{
        return 21.0f / (16.0f*M_PI*h*h*h);
	}

	static NVPARTICLES_CUDA_EXPORT
    float kernelVariable(const float h, const float r)
	{
        float q = r / h;
		if(q > 2.0f)
			return 0.0f;

        float val = 1.0f - 0.5f*q;
        val *= val;
        val *= val;
        val *= 1.0f + 2.0f*q;
		return val;
	}

	static NVPARTICLES_CUDA_EXPORT
    float gradientConstant(const float h)
	{
        return 105.0f / (128.0f*M_PI*h*h*h*h*h);
	}

	static NVPARTICLES_CUDA_EXPORT
    float gradientVariable(const float h, const float r)
	{
        const float qSub2 = r/h - 2.0f;
        float val = qSub2*qSub2*qSub2;
        return val;
	}
};

class Poly6
{
public:

	static NVPARTICLES_CUDA_EXPORT
    float kernelConstant(float h)
	{
		return 315.0f / (64.0f * M_PI * powf(h, 9.0f) );
	}

	static NVPARTICLES_CUDA_EXPORT
    float kernelVariable(float hPow2, float rPow2)
	{
		float hPow2_rPow2 = hPow2 - rPow2;
		return hPow2_rPow2 * hPow2_rPow2 * hPow2_rPow2;
	}

	static NVPARTICLES_CUDA_EXPORT
    float gradientConstant(float h)
	{
		return -945.0f / (32.0f * M_PI * powf(h, 9.0f) );
	}

	static NVPARTICLES_CUDA_EXPORT
    float gradientVariable(float h, float hPow2, float r)
	{
		float hPow2_rPow2 = hPow2 - (r*r);
		return hPow2_rPow2 * hPow2_rPow2;
	}

	static NVPARTICLES_CUDA_EXPORT
    float laplaceConstant(float h)
	{
		return 945.0f / (8.0f * M_PI * powf(h, 9.0f) );
	}

	static NVPARTICLES_CUDA_EXPORT
    float laplaceVariable(float h, float hPow2, float r, float rPow2)
	{
		float a = hPow2 - rPow2;
		float b = rPow2 - 0.75f*a;
		return a * b;
	}
};

class Cubic
{
public:

	static NVPARTICLES_CUDA_EXPORT
    float kernelConstant(float h)
	{
		return  1.0f / (M_PI * h*h*h);
	}

	static NVPARTICLES_CUDA_EXPORT
    float kernelVariable(float h, float r)
	{
		float q = r / h;

		if (q < 1.0f)
			return 1.0f - 1.5f*q*q + 0.75f*q*q*q;
		else
			return 0.25f*(2.0f - q)*(2.0f - q)*(2.0f - q);
	}

	static NVPARTICLES_CUDA_EXPORT
    float gradientConstant(float h)
	{
		return 3.f / ( 4.f * M_PI * h*h*h*h);
	}

	static NVPARTICLES_CUDA_EXPORT
    float gradientVariable(float h, float r)
	{
		float q = r / h;

		if(q < 1.0f)
			return (-4.0f + 3.0f*q)/h;
		else
			return -(q-2.0f)*(q-2.0f)/r;
	}

};

/// Viscosity kernel from Mller et al.
///
class Viscosity
{
public:

	static NVPARTICLES_CUDA_EXPORT
    float kernelConstant(float h)
	{
		return 15.0f / (M_PI * powf(h, 6.0f) );
	}

	static NVPARTICLES_CUDA_EXPORT
    float kernelVariable(float h, float r)
	{
		float h_r =  (h-r);
		return h_r * h_r * h_r;
	}

	static NVPARTICLES_CUDA_EXPORT
    float gradientConstant(float h)
	{
		return 15.0f / (2*M_PI * h*h*h);
	}

	static NVPARTICLES_CUDA_EXPORT
    float gradientVariable(float h, float r)
	{
		float a = (-3*r) / (2*h*h*h);
		float b = (2 / h*h);
		float c = -h / (2*r*r*r);
		return a + b + c;
	}

	static NVPARTICLES_CUDA_EXPORT
    float laplaceConstant(float h)
	{
		return 45.0f / (M_PI * powf(h, 6.0f) );
	}

	static NVPARTICLES_CUDA_EXPORT
    float laplaceVariable(float h, float rlen)
	{
		float h_rlen = (h-rlen);
		return h_rlen;
	}
};

class Spiky
{
public:

	static NVPARTICLES_CUDA_EXPORT
    float kernelConstant(float h)
	{
		return 15.0f / (M_PI * powf(h, 6.0f) );
	}

	static NVPARTICLES_CUDA_EXPORT
    float kernelVariable(float h, float r)
	{
		float h_r =  (h - r);
		return h_r*h_r*h_r;
	}

	static NVPARTICLES_CUDA_EXPORT
    float gradientConstant(float h)
	{
		return -45.0f / (M_PI * powf(h, 6.0f) );
	}

	static NVPARTICLES_CUDA_EXPORT
    float gradientVariable(float h, float r)
	{
		float h_r = (h-r);
		return (1.0f/r)*(h_r*h_r);
	}

	static NVPARTICLES_CUDA_EXPORT
    float laplaceConstant(float h)
	{
		return -90.0f / (M_PI * powf(h, 6.0f) );
	}

	static NVPARTICLES_CUDA_EXPORT
    float3 laplaceVariable(float h, float r, float3 rPos)
	{
		float h_r = (h-r);
		float h_2r = (h-2.0f*r);
		return (1.0f/rPos) * (h_r*h_2r);
	}
};

}
#endif
